/***********************************************************************/
/*                                                                     */
/*                          ANALOGICAL TRANSFERENCE                    */
/*                Copyright (C) 1993 Eric Charles Steinhart            */
/*                            All rights reserved.                     */
/*                                                                     */
/***********************************************************************/
#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\string.h"
#include "anadefs.h"
#include "semnet.h"
#include "map.h"
#include "buffer.h"
#include "iterate.h"


struct SEMANTIC_NETWORK *AnalogicalTransference( S, map, topic, source)
struct SEMANTIC_NETWORK *S;
struct map_structure *map;
cons_ptr topic;
cons_ptr source;
{
	cons_ptr SourceFrames, SourceRules;
	cons_ptr SourceNouns;
	cons_ptr NewTopicFrames;
	cons_ptr NewTopicTerms;
	cons_ptr ptr, tptr;
	cons_ptr SFP, SRP;
	cons_ptr SourceDescriptions, SDP;
	int SD, TD, SF, TF, SR, TR;
	int TopicAntecedent, TopicConsequent, NewConj, NewMF;
	int AntecedentHasAnalogue, AllConsequentsHaveAnalogues;
	int TheConsequent, ThisSrcConseq;
	cons_ptr SNFeatures, TNFeatures;
	cons_ptr SourceMFrames, SMF;
	int SN, TN;
	int i;
	int position;
	int NodeType;
	char NodeName[MAX_STRLEN];
	char suffix[MAX_STRLEN];
	cons_ptr SourceProperties;
	int SourceObject, TopicObject;
	cons_ptr SourceValues;
	int ThisValue;
	struct TextBuffer *FB;

	/***********************************************************************/
	/*                     TRANSFER PROPERTIES TO TOPIC                    */
	/***********************************************************************/
	DisplayExtraMsg( FieldChoiceWindow,
		"Performing Analogical Transference: Moving source properties      ");
	/* SourceProperties = { x in Source | x is a PROPERTYTYPE } */
	SourceProperties = NIL;
	ListIterator( ptr, source) {
		if (GetType(S,car(ptr)) == PROPERTYTYPE)
			SourceProperties = cons(car(ptr), SourceProperties);
		}
/*  printf("The SourceProperties are:\n");
	print_node_names( SourceProperties, S);
	strike_any_key();*/

	ListIterator( SFP, SourceProperties) {
		int SourceAttr;

		SF = car(SFP); /* SF is the source property */
/*      printf("Processing SourceProperty <%s>.\n", GetName( S,SF));*/

		SourceAttr = GetAttributeTerm(S, SF);

		/* add a new target property to the semantic net */
		NodeType = PROPERTYTYPE;
		strcpy(NodeName, "PID");
		itoa(GetNumNodes( S) + 1, suffix, 10);
		strcat(NodeName, suffix);
		TF = GetNumNodes( S); /* the new node is the last node */
		S = AddNode(S, NodeType, NodeName);
		topic = cons(TF, topic);
		NewTopicFrames = cons(TF, NewTopicFrames);

		/* add (TF, SF) to analogical map */
		map = TransferPairToMap( map, TF, SF);

		/* add connections between new topic property and attribute */
		topic = cons( SourceAttr, topic);
		NewTopicTerms = cons(SourceAttr, NewTopicTerms);
		S = AddConnections(S, TF, SourceAttr, ATTRIBUTE, ANALOGICAL, 0);

		/* add connections between new topic property and object */
		SourceObject = GetObjectTerm(S, SF);
		TopicObject = get_analogue(S, map, SourceObject);
		/* WARNING: what if there is no TopicObject?  */
		/* Doing this after transfering frames        */
		/* minimizes but does not eliminate the risk. */
		S = AddOneConnection(S, TF, TopicObject, OBJECT, ANALOGICAL);
		S = AddOneConnection(S, TopicObject, TF, PROPERTY, ANALOGICAL);

		/* for each value V of the source property, */
		/*     add ANALOGICAL-value connections from TF to V */
		SourceValues = NIL;
		SourceValues = GetLiteralValues(S, SF, SourceValues);
/*      printf("The SourceValues are:\n");
		print_node_names( SourceValues, S);
		strike_any_key();*/
		ListIterator( ptr, SourceValues) {
			/* add connections between new topic property and value */
			ThisValue = car(ptr);
/*          printf("Process value: <%s>\n", GetName( S,ThisValue));*/
			topic = cons( ThisValue, topic);
			NewTopicTerms = cons(ThisValue, NewTopicTerms);
			S = AddConnections(S, TF, ThisValue, VALUE, ANALOGICAL, 0);
			}
		FreeList( SourceValues);
		} /* end for each SF in SourceProps */
	FreeList( SourceProperties);


	/***********************************************************************/
	/*                  TRANSFER DESCRIPTIONS TO TOPIC                     */
	/***********************************************************************/
	DisplayExtraMsg( FieldChoiceWindow,
	  "Performing Analogical Transference: Moving source descriptions      ");
	/* SourceDescriptions = { x in Source | x is a DESCRIPTION } */
	SourceDescriptions = NIL;
	ListIterator( ptr, source) {
		if (GetType(S,car(ptr)) == DESCRIPTION)
			SourceDescriptions = cons(car(ptr), SourceDescriptions);
		}
/*  printf("The SourceDescriptions are:\n");
	print_node_names( SourceDescriptions, S);
	strike_any_key();*/

	ListIterator( SDP, SourceDescriptions) {
		int T; /* ranges over surface positions */

		SD = car(SDP); /* SD is the source description */
/*      printf("Processing SourceDescription <%s>.\n", GetName(S,SD));*/

		if (not(get_analogue(S, map, SD))) {
			/* add a new target description to the semantic net */
			NodeType = DESCRIPTION;
			strcpy(NodeName, "DID");
			itoa(GetNumNodes(S) + 1, suffix, 10);
			strcat(NodeName, suffix);
			TD = GetNumNodes( S); /* the new node is the last node */
			S = AddNode(S, NodeType, NodeName);
			topic = cons(TD, topic);
			NewTopicFrames = cons(TD, NewTopicFrames);

			/* add (TD, SD) to analogical map */
			map = TransferPairToMap( map, TD, SD);

			/* add connections between new topic frame and nouns */
			for (position = 1; position < MAX_SURFACE_POSITIONS; position++) {
				int SourceNoun, TargetNoun;
				int connection, role;
				SourceNoun = GetHeadAtPosition(S, SD, position);

				if (SourceNoun != NO_FILLER) {
/*                  printf("Source desc <%s> has noun <%s> in pos <%d>.\n",
						GetName(S,SD), GetName(S,SourceNoun), position);*/

					connection = GetConn(S, SourceNoun, SD);
					role = GetRole(S, connection);

					TargetNoun = get_analogue(S, map, SourceNoun);
/*                  printf("The analogue of <%s> is <%s>.\n",
							GetName(S,SourceNoun), GetName(S,TargetNoun));*/

					if (TargetNoun != NIL) {
						S = AddConnections(
								S, TD, TargetNoun, role, LITERAL, position);
						}
					else {
						TargetNoun = SourceNoun;
						topic = cons(TargetNoun, topic);
						NewTopicTerms = cons(TargetNoun, NewTopicTerms);
						S = AddConnections(
								S, TD, TargetNoun, role, ANALOGICAL, position);
						}
					} /* end if not null SourceNoun */
				} /* end for each surface position */
			} /* end if is_member SourceDescription in topic */
		} /* end for each SD in SourceDescriptions */
		FreeList( SourceDescriptions);


	/***********************************************************************/
	/*                     TRANSFER FRAMES TO TOPIC                        */
	/***********************************************************************/
	DisplayExtraMsg( FieldChoiceWindow,
	  "Performing Analogical Transference: Moving source sentences         ");
	NewTopicFrames = NIL;
	NewTopicTerms = NIL;

	/* SourceFrames = { x in Source | x is an SFRAME } */
	SourceFrames = NIL;
	ListIterator( ptr, source) {
		if (GetType(S,car(ptr)) == SFRAME)
			SourceFrames = cons(car(ptr), SourceFrames);
		}
/*  printf("The SourceFrames are:\n");
	print_node_names( SourceFrames, S);
	strike_any_key();*/

	ListIterator( SFP, SourceFrames) {
		int SourceVerb;
		int T; /* ranges over thematic roles */

		SF = car(SFP); /* SF is the source frame */
/*      printf("Processing SourceFrame <%s>.\n", GetName( S,SF));*/

		SourceVerb = GetFrameVerb(S, SF);
		if (SourceVerb == NO_FILLER) {
			printf("ERROR: Source frame <%s> has no verb.\n",
					GetName( S,SourceVerb));
			exit( ERROR_VALUE);
			}

		if (not(get_analogue(S, map, SF))) {
			/* add a new target frame to the semantic net */
			NodeType = SFRAME;
			strcpy(NodeName, "FID");
			itoa(GetNumNodes( S) + 1, suffix, 10);
			strcat(NodeName, suffix);
			TF = GetNumNodes( S); /* the new node is the last node */
			S = AddNode(S, NodeType, NodeName);
			topic = cons(TF, topic);
			NewTopicFrames = cons(TF, NewTopicFrames);

			/* add (TF, SF) to analogical map */
			map = TransferPairToMap( map, TF, SF);

			/* add connections between new topic frame and verb */
			topic = cons( SourceVerb, topic);
			NewTopicTerms = cons(SourceVerb, NewTopicTerms);
			S = AddConnections(S, TF, SourceVerb, VERB, ANALOGICAL, 2);

			/* add connections between new topic frame and nouns */
			for (T = 1; T < MAX_FUNC_ROLES; T++) {
				cons_ptr TheNouns;
				int x, y;
				int connection, position;

				/* find the nouns that play thematic role T in frame SF */
				TheNouns = NIL;
				TheNouns = node_trans( SF, S, T, LITERAL, TheNouns);
				if (length(TheNouns) > 1) {
					printf(
					 "ERROR: Source frame <%s> has many nouns for role <%s>.\n",
					 GetName( S,SF), role_names[T]);
					exit(ERROR_VALUE);
					}

				if (not(null(TheNouns))) {
					x = car(TheNouns);
/*                  printf("Source frame <%s> has noun <%s> for role <%s>.\n",
						GetName( S,SF), GetName( S,x), role_names[T]);*/
					y = get_analogue(S, map, x);
/*                  printf("The analogue of <%s> is <%s>.\n",
							GetName( S,x), GetName( S,y));*/
					connection = GetConn(S, x, SF);
					position = GetSurfPos(S,connection);
					if (not(y == NIL)) {
						S = AddConnections(S, TF, y, T, LITERAL, position);
						}
					if (y == NIL) {
						/* this is how terms like "domain" get transferred */
						/* in THE BRAIN IS A FERROMAGNET analogy */
						y = x; /* just transfer the source noun */
							   /* NOTE that the source noun is ANALOGICAL*/
						topic = cons(y, topic);
						NewTopicTerms = cons(y, NewTopicTerms);
						S = AddConnections(S, TF, y, T, ANALOGICAL, position);
						}
					} /* end if not null x */
				} /* end for each thematic role */
			} /* end if is_member SourceFrame in topic */
		} /* end for each SF in SourceFrames */
	FreeList( SourceFrames);


	/***********************************************************************/
	/*                    TRANSFER FEATURES TO TOPIC                       */
	/***********************************************************************/
	DisplayExtraMsg( FieldChoiceWindow,
	  "Performing Analogical Transference: Moving source features          ");
	SourceMFrames = NIL;
	/* SourceMFrames = { x in Source | MFRAME(x) } */
	ListIterator( ptr, source) {
		if (GetType(S,car(ptr)) == MFRAME)
			SourceMFrames = cons(car(ptr), SourceMFrames);
		}

	ListIterator( SMF, SourceMFrames) {
		/* for each MFRAME SMF in Source do */

		cons_ptr SourceNounList, FeaturesOfFrame, TNMFrames;
		int ThisSrcFrame, SourceNoun, SourceAdj;
		int TargetNoun, ThisFrame, opposite;

		ThisSrcFrame = car(SMF);

		/* get the SUBJECT of ThisSrcFrame into SourceNoun */
		SourceNounList = NIL;
		SourceNounList = node_trans(ThisSrcFrame, S,
									SUBJECT, LITERAL, SourceNounList);
		SourceNoun = car(SourceNounList);
		FreeList( SourceNounList);

		/* get the PREDICATE of ThisSrcFrame into SourceAdj */
		FeaturesOfFrame = NIL;
		FeaturesOfFrame = node_trans(ThisSrcFrame, S,
									PREDICATE, LITERAL, FeaturesOfFrame);
		SourceAdj = car(FeaturesOfFrame);
		FreeList( FeaturesOfFrame);

/*      printf("Transfering source feature <%s(%s)>.\n",
				GetName(S, SourceAdj), GetName(S, SourceNoun));*/

		TargetNoun = get_analogue(S, map, SourceNoun);
		if (not(TargetNoun == NIL)) {
/*          printf("Source noun <%s> has analogue <%s>.\n",
					GetName(S, SourceNoun), GetName(S, TargetNoun));*/

			/* get the features of TargetNoun into TNFeatures */
			TNFeatures = NIL;
			TNFeatures = GetFeatures( S, TargetNoun, LITERAL, TNFeatures);
/*          printf("The features of <%s> are:\n", GetName(S, TargetNoun));
			print_node_names( TNFeatures, S);*/

			opposite = FindOpposite(S, SourceAdj);
			/* Negative Analogies are filtered out. */
			if (not(is_member(opposite, TNFeatures))) {
				/* add the source adjective to the Topic Field */
/*              printf("Add <%s> to target.\n", GetName(S, SourceAdj));*/

				topic = cons( SourceAdj, topic);
				NewTopicTerms = cons( SourceAdj, NewTopicTerms);

				/* make a new modifier frame (MFRAME) node */
				NodeType = MFRAME;
				strcpy(NodeName, "MF");
				itoa(GetNumNodes( S), suffix, 10);
				strcat(NodeName, suffix);
				NewMF = GetNumNodes( S); /* the new node is the last node */
				S = AddNode(S, NodeType, NodeName);
				topic = cons(NewMF, topic);
				NewTopicFrames = cons(NewMF, NewTopicFrames);
/*              printf("Made new target MFRAME: <%s>.\n",GetName(S,NewMF));*/
				/* add (NewMF, ThisSrcFrame) to analogical map */
				map = TransferPairToMap( map, NewMF, ThisSrcFrame);

				/* add the new connections */
				S = AddOneConnection(S, TargetNoun, NewMF,
											SUBJECT, ANALOGICAL);
				S = AddOneConnection(S, NewMF, TargetNoun,
											SUBJECT, ANALOGICAL);
				S = AddOneConnection(S, SourceAdj, NewMF,
											PREDICATE, ANALOGICAL);
				S = AddOneConnection(S, NewMF, SourceAdj,
											PREDICATE, ANALOGICAL);
				} /* if not is_member */
				FreeList( TNFeatures);
			} /* end if not null TargetNoun */
		} /* end for each SMF in SourceMFrames */
	FreeList( SourceMFrames);

	/***********************************************************************/
	/*                      TRANSFER RULES TO TOPIC                        */
	/***********************************************************************/
	DisplayExtraMsg( FieldChoiceWindow,
	  "Performing Analogical Transference: Moving source rules             ");
	/* SourceRules = { x in Source | x is a RULE } */
	SourceRules = NIL;
	ListIterator( ptr, source) {
		if (GetType(S,car(ptr)) == RULE)
			SourceRules = cons(car(ptr), SourceRules);
		}

	ListIterator( SRP, SourceRules) {
		/* for every source rule do */

		int SourceAntecedent;
		cons_ptr SourceConsequents;

		SR = car(SRP); /* SR is the source rule */
/*      printf("Processing SourceRule <%s>.\n", GetName( S,SR));*/

		SourceAntecedent = GetAntecedent(S, SR);
/*      printf("SourceAntecedent is <%s>\n", GetName( S,SourceAntecedent));*/

		SourceConsequents = NIL;
		SourceConsequents = GetAllConsequentFrames(S, SR, SourceConsequents);
/*      printf("SourceConsequents are:\n");
		print_node_names( SourceConsequents, S);*/

		AntecedentHasAnalogue
			= AllHaveAnalogues(S, map, ncons(SourceAntecedent));
/*      if (AntecedentHasAnalogue)
			printf("Src Antecedent has analogue.\n");
		else printf("Src Antecedent has no analogue.\n");*/

		AllConsequentsHaveAnalogues
			= AllHaveAnalogues(S, map, SourceConsequents);
/*      if (AllConsequentsHaveAnalogues)
			printf("All Src Consequents have analogues.\n");
		else printf("NOT all Src Consequents have analogues.\n");*/

/*      ListIterator( ptr, SourceConsequents) {
			int AnaTemp;
			AnaTemp = get_analogue(S, map, car(ptr));
			if (AnaTemp != NIL)
				printf("SrcConseq <%s> has analogue <%s>.\n",
						GetName(S,car(ptr)), GetName(S,AnaTemp));
			else printf("SrcConseq <%s> has NO analogue.\n",
						GetName(S,car(ptr)));
			}*/

		if (AntecedentHasAnalogue && AllConsequentsHaveAnalogues) {
			/* then transfer the source rule SR as follows */

			/* make a new target rule node TR */
			NodeType = RULE;
			strcpy(NodeName, "RID");
			itoa(GetNumNodes( S), suffix, 10);
			strcat(NodeName, suffix);
			TR = GetNumNodes( S); /* the new node is the last node */
			S = AddNode(S, NodeType, NodeName);
			topic = cons(TR, topic);
			NewTopicFrames = cons(TR, NewTopicFrames);
/*          printf("Making new topic rule <%s>.\n", NodeName);*/

			/* add (TR, SR) to analogical map */
			map = TransferPairToMap( map, TR, SR);

			/* link TR to the transferred antecedent */
			TopicAntecedent = get_analogue(S, map, SourceAntecedent);
/*          printf("TopicAntecedent is <%s>.\n",GetName(S,TopicAntecedent));*/

			/* add connections between new topic rule & antecedent */
			S = AddConnections(S, TR, TopicAntecedent,
										ANTECEDENT, ANALOGICAL, 0);

			TheConsequent = GetConsequent(S, SR);
			if (GetType(S,TheConsequent) != CONJUNCTION) {
				/* then SR has an SFRAME as its consequent */
				/* link TR to the analogue of the consequent of SR */
				TopicConsequent = get_analogue(S, map, TheConsequent);
				S = AddConnections(S, TR, TopicConsequent,
										CONSEQUENT, ANALOGICAL, 0);
				}
			else { /* SR has a CONJUNCTION as its consequent */
				/* make a new conjunction node NewConj */
				NodeType = CONJUNCTION;
				strcpy(NodeName, "CONJ");
				itoa(GetNumNodes( S), suffix, 10);
				strcat(NodeName, suffix);
				NewConj = GetNumNodes( S); /* the new node is the last node */
				S = AddNode(S, NodeType, NodeName);
				topic = cons(NewConj, topic);
				NewTopicFrames = cons(NewConj, NewTopicFrames);

				/* add (NewConj, TheConsequent) to analogical map */
				map = TransferPairToMap( map, NewConj, TheConsequent);

				/* link TR to the NewConj as a consequent */
				S = AddConnections(S, TR, NewConj,
										CONSEQUENT, ANALOGICAL, 0);

				/* for each M in SourceConsequents, */
				/*      link the target analogue of M to NewConj */
				ListIterator( tptr, SourceConsequents) {
					ThisSrcConseq = car(tptr);
					TopicConsequent = get_analogue(S, map, ThisSrcConseq);
/*                  printf("Link <%s> to <%s> as ANDBRANCH\n",
						GetName( S,NewConj),
						GetName( S,TopicConsequent));*/
					S = AddConnections(S, NewConj, TopicConsequent,
										ANDBRANCH, ANALOGICAL, 0);
					}
				} /* end else SR has a CONJUNCTION as its consequent */
			} /* if antecedent & all consequents have analogues */
/*          strike_any_key();*/
		} /* for every source rule */
	FreeList( SourceRules);

/*  printf("The new frames added to the topic are:\n");
	print_node_names( NewTopicFrames, S);
	printf("The new terms added to the topic are:\n");
	print_node_names( NewTopicTerms, S);
	strike_any_key();*/
	FreeList( NewTopicFrames);
	FreeList( NewTopicTerms);

	/* fill the FieldBuffer with the new topic field */
	/* initialize the FieldBuffer */
	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);
	StringMove( FB, "------- Please Scroll Down -------");
	NextLine( FB);
	StringMove( FB, "The new topic field is: ");
	NextLine( FB);
	FillFieldBuffer( S, FB, nconseq(topic));
	StringMove( FB, "***** End of Text *****");
	NextLine( FB);
	_clearscreen( _GCLEARSCREEN);
	DisplayBuffer( FB, FieldScrollWindow);
	free( FB);

	return( S); /* return modified semantic net */
}



/***************************************************************************/
/*                                                                         */
/*          Metaphorical Redescription via Analogical Transference         */
/*                                                                         */
/***************************************************************************/
void metaphorical_redescription( S)
struct SEMANTIC_NETWORK *S;
{
	int i;
	int target_cue;                 /* the cue */
	int target_type;                /* type of target field */
	int need_target, ThereIsAnError;
	char ErrorMsg[80];
	sequence_ptr cand_targs;        /* candidate target fields */
	cons_ptr target;                /* the target field */
	sequence_ptr cand_srcs;         /* the candidate source fields */
	cons_ptr source;                /* the source field */
	struct map_structure *map;
	struct TextBuffer *FB;

	_clearscreen( _GCLEARSCREEN);

	/*******************************************************************/
	/*            Generate, Display, and Choose Target Field           */
	/*******************************************************************/
	/* generate candidate target fields from the cue */
	need_target = TRUE;
	ThereIsAnError = FALSE;
	while( need_target) {
		_clearscreen( _GCLEARSCREEN);
		DisplayTermsInNetwork( S, TermsWindow);
		DisplayWindow( TopicNodeQueryWindow);
		DisplayWindow( ErrorWindow);
		if (ThereIsAnError) PrintError( ErrorMsg);
		/* get the target cue */
		target_cue = query_user_for_node(S, TopicNodeQueryWindow);
		_clearscreen( _GCLEARSCREEN);
		/* generate target field from the cue */
		cand_targs = NIL;
		target_type = fields_type_menu( S, target_cue);
		cand_targs = get_fields(S, target_cue, target_type, cand_targs);
		if (nullseq( cand_targs)) {
			ThereIsAnError = TRUE;
			sprintf(ErrorMsg, "Target field for <%s> was NIL.",
					GetName( S,target_cue));
			}
		else need_target = FALSE;
		}

	/* fill the FieldBuffer with the candidate target fields */
	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);
	StringMove( FB, "------- Please Scroll Down -------");
	NextLine( FB);
	StringMove( FB, "The target fields are: ");
	NextLine( FB);
	FillFieldBuffer( S, FB, cand_targs);
	StringMove( FB, "***** End of Text *****");
	NextLine( FB);
	DisplayBuffer( FB, FieldScrollWindow);

	/* select a target field for access */
	if (lengthseq( cand_targs) > 1) {
		/* need to select a source */
		DisplayWindow( FieldChoiceWindow);
		target = NIL;
		target = select_one_field( cand_targs, target, FieldChoiceWindow);
		}
	else target = carseq( cand_targs);


	/*******************************************************************/
	/*            Generate, Display, and Choose Source Field           */
	/*******************************************************************/
	/* generate candidate source fields */
	cand_srcs = NIL;
	cand_srcs = gen_cand_srcs( S, target, target_type, cand_srcs);
	if (nullseq( cand_srcs)) {
		printf("ERROR: No source fields.\n");
		return;
		}

	/* fill the FieldBuffer with the candidate source fields */
	InitializeTextBuffer( FB);
	StringMove( FB, "------- Please Scroll Down -------");
	NextLine( FB);
	StringMove( FB, "The source fields are: ");
	NextLine( FB);
	FillFieldBuffer( S, FB, cand_srcs);
	StringMove( FB, "***** End of Text *****");
	NextLine( FB);
	DisplayBuffer( FB, FieldScrollWindow);

	/* select one source field for access */
	if (lengthseq( cand_srcs) > 1) {
		/* need to select a source */
		DisplayWindow( FieldChoiceWindow);
		source = NIL;
		source = select_one_field( cand_srcs, source, FieldChoiceWindow);
		}
	else source = carseq( cand_srcs);


	/*******************************************************************/
	/*                    Generate an Analogical Map                   */
	/*******************************************************************/
	map = AllocateMap();
	/* make_map_from_net() uses a hypothesis net to make a map */
	map = make_map_from_net( S, map, target, source);

	/*******************************************************************/
	/*                   Perform ANALOGICAL TRANSFERENCE               */
	/*******************************************************************/
	if (GetMapLength(map) == NIL) {
		printf("No analogies between target and source.\n");
		printf("Analogical transference is not possible.\n");
		return;
		}
	DisplayWindow( FieldChoiceWindow);
	DisplayExtraMsg( FieldChoiceWindow,
		"Performing Analogical Transference.....                       ");
	S = AnalogicalTransference( S, map, target, source);

	/* interpret the analogically transferred frames? */
	DisplayWindow( FieldChoiceWindow);
	QueryAndResponse( FieldChoiceWindow,
		"Interpret propositions? (Y or N):        [  ]");
	if (GetYesOrNoFromWindow( FieldChoiceWindow) == YES)
		InterpretTransferredFrame(S, map);
	free( map);
	free( FB);
}
