/***********************************************************************/
/*                                                                     */
/*                              SEMANTIC FIELDS                        */
/*                Copyright (C) 1992 Eric Charles Steinhart            */
/*                            All rights reserved.                     */
/*                                                                     */
/***********************************************************************/

#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "anadefs.h"
#include "semnet.h"
#include "buffer.h"
#include "iterate.h"

/***********************************************************************/
/*                                                                     */
/*                        Functional Semantic Fields                   */
/*                                                                     */
/***********************************************************************/

/* compute the set Z(n) for a node n */
cons_ptr Z( S, n, result)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
int n;                          /* a node */
cons_ptr result;                /* a list of nodes that is Z(n) */
{
	cons_ptr pterms, qterms, s;
	int y; /* a node */

	/* compute A2(n, LITERAL) */
	qterms = NIL;
	qterms = gen_node_trans(n, S, LITERAL, qterms);
	pterms = NIL;
	pterms = gen_state_trans(qterms, S, LITERAL, pterms);

	s = pterms;
	while( not( null( s))) {
		y = car(s);
		if (GetType( S,y) == NOUN)
			result = cons( y, result);
		s = cdr(s); /* get next s */
		}
	return( result);
}


/* computes the similarity of two terms */
float sim( S, x, y)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
int x;                          /* the first term */
int y;                          /* the second term */
{
	cons_ptr ptermsx, ptermsy, mutual;
	float s;
	void strike_any_key( void);

/*      printf("Computing sim( %s, %s)\n",
			GetName( S, x), GetName( S, y)); */
	/* compute A1 of x */
	ptermsx = NIL;
	ptermsx = gen_node_trans( x, S, LITERAL, ptermsx);
	/* compute A1 of y */
	ptermsy = NIL;
	ptermsy = gen_node_trans( y, S, LITERAL, ptermsy);

	/* compute the similarity */
	mutual = NIL;
	mutual = intersect( ptermsx, ptermsy, mutual);
/*      printf("Intersection is: "); print_list( mutual); printf("\n"); */
/*      printf("Length( intersection) = %d\n", length( mutual)); */
/*      printf("Length( A1(x)) = %d\n", length( ptermsx)); */
		s = (float) length( mutual) / (float) length( ptermsx);
/*      printf("Sim( %s, %s) = %f\n",
		GetName( S, x), GetName( S, y), s); */
/*      strike_any_key(); */  /* pause for debug */
	return( s);
}

/* compute the cluster of densely interconnected nouns in which a */
/* particular noun x sits */
cons_ptr cluster( S, x, Zset, result)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
int x;                          /* the noun */
cons_ptr Zset;                  /* the set of nouns to which x belongs */
cons_ptr result;                /* the cluster in which x sits */
{
	cons_ptr t;
	int y;

	t = Zset;
	while( not( null( t))) {
		y = car(t);
		if (sim(S,y,x) > 0.5)
			result = cons( y, result);
		t= cdr(t);  /* get next t */
		}
	return( result);
}


int NoContainingCluster( the_cluster, partitions)
cons_ptr the_cluster;
sequence_ptr partitions;
{
	sequence_ptr PartitionsPtr;
	cons_ptr ClusterInPartitions;

	PartitionsPtr = partitions;
	while (not( nullseq( PartitionsPtr))) {
		ClusterInPartitions = carseq( PartitionsPtr);
		if ((is_subset( the_cluster, ClusterInPartitions))
			&& not( equal( the_cluster, ClusterInPartitions)))
			return( FALSE);
		PartitionsPtr = cdrseq( PartitionsPtr);
		}
	return( TRUE);
}


/* Find the clusters of densely interconnected nouns in Zset */
sequence_ptr ClusterSet( S, Zset, result)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
cons_ptr Zset;                  /* the set to partition */
sequence_ptr result;            /* the resultant partitions of Zset */
{
	cons_ptr t, the_cluster;
	int y;

	t = Zset;
	while( not( null( t))) {
		y = car(t);
		the_cluster = NIL;
		the_cluster = cluster( S, y, Zset, the_cluster);
/*      printf("The cluster of %s is:\n", GetName( S, y));
		print_node_names( the_cluster, S);*/
		result = unionseq( nconseq( the_cluster), result);
		t = cdr(t); /* get next noun */
		}
	return( result);
}

/* Find the partitions of densely interconnected nouns in Zset */
sequence_ptr partitions( S, Zset, result)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
cons_ptr Zset;                  /* the set to partition */
sequence_ptr result;            /* the resultant partitions of Zset */
{

	sequence_ptr TheSetOfClusters;
	sequence_ptr Ptr;
	cons_ptr TheCluster;

/*  printf("ENTER:partitions.\n");*/
/*  printf("In partitions: Zset = ");
	print_node_names( Zset, S); printf("\n");
	strike_any_key();*/

	TheSetOfClusters = NIL;
	TheSetOfClusters = ClusterSet(S, Zset, TheSetOfClusters);

	Ptr = TheSetOfClusters;
	while( not( nullseq( Ptr))) {
		TheCluster = carseq(Ptr);
/*      printf("In partitions: TheCluster = ");
		print_node_names( TheCluster, S); printf("\n");
		strike_any_key();*/
		if (NoContainingCluster(TheCluster, TheSetOfClusters)) {
/*          printf("No containing clusters.\n");*/
			result = unionseq( nconseq(TheCluster), result);
			}
		Ptr = cdrseq(Ptr); /* get next cluster */
		}
/*      printf("EXIT:partitions\n");*/
	return( result);
}

/* finds the paradigmatic closure of a field by adding all        */
/* paradigmatic frames to the field                               */
/*      for each x in field do {                                  */
/*          if (is_adjective(x)) {                                */
/*             ParaFrames = { P in S | type(P) = PFRAME & x in P} */
/*             field = field UNION { y | y in P}                  */
/*             }                                                  */
/*      }                                                         */
cons_ptr ParadigmaticClosure( S, field)
struct SEMANTIC_NETWORK *S;
cons_ptr field;
{
	cons_ptr FieldPtr;
	int Term;
	cons_ptr ParaFrames, ParaTerms;

/*  printf("ENTER: ParadigmaticClosure\n");*/
	FieldPtr = field;
	while( not( null( FieldPtr))) {
		Term = car(FieldPtr);
		if (is_adjective(S,Term)) {
/*                      printf("Processing adjective: <%s>\n",
					GetName( S, Term));*/
			ParaFrames = NIL;
			ParaFrames = para_node_trans( Term, S,
						LITERAL, ParaFrames);
			if (length(ParaFrames) > 1) {
			   printf(
			      "ERROR:<%s> is in many paradigmatic frames.\n",
					GetName( S, Term));
			   exit(ERROR_VALUE);
			   }
			ParaTerms = NIL;
			ParaTerms = para_node_trans(first(ParaFrames), S,
					LITERAL, ParaTerms);
			ParaTerms = list_union( ParaFrames, ParaTerms);
			field = list_union( ParaTerms, field);
			}
		FieldPtr = cdr(FieldPtr);
		}
/*  printf("EXIT: ParadigmaticClosure\n");*/
	return( field);
}


cons_ptr FeatureClosure( S, field)
struct SEMANTIC_NETWORK *S;
cons_ptr field;
{
	int node;
	cons_ptr t, NodeFeatures;

	ListIterator( t, field) {
		node = car(t);
		if (GetType( S,node) == NOUN) {
			NodeFeatures = NIL;
			NodeFeatures = node_trans(
							node, S, SUBJECT, LITERAL, NodeFeatures);
			field = list_union( NodeFeatures, field);
			}
		}
	return( field);
}

sequence_ptr ExplicitFieldsOf(S, term, result)
struct SEMANTIC_NETWORK *S;
int term;
sequence_ptr result;
{
	sequence_ptr ExpFieldPtr;
	cons_ptr ThisExpField;

	ExpFieldPtr = GetFields( S);
	while(ExpFieldPtr != NIL) {
		ThisExpField = carseq( ExpFieldPtr);
		if (is_member(term, ThisExpField))
			result = conseq(ThisExpField, result);
		ExpFieldPtr = cdrseq( ExpFieldPtr);
		}
	return( result);
}

/* find the functional fields in which a node sits */
sequence_ptr func_fields( S, n, result)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
int n;                          /* the node */
sequence_ptr result;            /* the fields in which n sits */
{
	sequence_ptr generators; /* clusters of nouns */
	sequence_ptr t;
	cons_ptr Zset;           /* used to store Z(n) */
	cons_ptr C;              /* a particular cluster */
	cons_ptr pframes, pterms, A2;

	/* check first for explicit semantic fields */
	/* if they exist, they override other generation methods */
	if (GetFields( S) != NIL) {
		/* then there are explicit semantic fields */
		result = ExplicitFieldsOf(S, n, result);
		/* if there are explicit semantic fields for the term */
		/* then return them */
		/* else use regular generation methods */
		if (not(nullseq(result))) return( result);
		}

	/* find Z(n) */
	Zset = NIL;
	Zset = Z( S, n, Zset);
	_clearscreen( _GCLEARSCREEN);
/*  printf("The set Z(%s) is:\n", GetName( S, n));
	print_node_names(Zset, S);
	strike_any_key();*/  /* pause for debugging */

	/* find the set of clusters in Z(n) */
	generators = NIL;
	generators = partitions( S, Zset, generators);
/*  printf("The clusters of Z(%s) are:\n", GetName( S, n));
	print_sequence_names( generators, S);
	strike_any_key();*/  /* pause for debugging */

	/* for every cluster C in partitions(Z(n)) */
	t = generators;
	while( not( nullseq( t))) {
		C = carseq( t);
/*      printf("Find the field generated by:\n");
		print_node_names( C, S);
		strike_any_key();*/  /* pause for debugging */
		/* find the field generated by C */
		/* find the participating frames */
		pframes = NIL;
		pframes = gen_state_trans(C, S, LITERAL, pframes);
		/* find the participating terms */
		pterms = NIL;
		pterms = gen_state_trans(pframes, S, LITERAL, pterms);
		/* field is the union of pframes & pterms */
		A2 = NIL;
		A2 = list_union( pframes, pterms);
		/* add paradigmatic frames to the field */
		A2 = ParadigmaticClosure( S, A2);
		/* add features to the field */
		A2 = FeatureClosure( S, A2);
		result = unionseq( nconseq( A2), result);
		t = cdrseq( t); /* get next cluster */
		}
	return( result);
}


/***********************************************************************/
/*                     Mereological Closure Algorithms                 */
/*                                                                     */
/* Algorithms for computing the mereological closure of a field by the */
/* method of least maximal containing wholes.                          */
/***********************************************************************/

static int Index[MAX_NODES];

/* find the nouns in a semantic field */
cons_ptr NounsInField( S, F)
struct SEMANTIC_NETWORK *S;
cons_ptr F;     /* the semantic field */
{
	cons_ptr NF, fptr;
	int x;

/*  printf("ENTER: NounsInField\n");*/
	NF = NIL;
	/* for each x in F, if x is a noun, add it to NF */
	fptr = F;
	while (not (null( fptr))) {
		x = car(fptr);
		if (GetType( S,x) == NOUN) NF = cons( x, NF);
		fptr = cdr( fptr);
		}
/*  print_node_names( NF, S);
	printf("EXIT: NounsInField\n");*/

	return( NF);
}

/* climb up the mereological chain of x to find the */
/* least maximal containing whole of x */
int LeastMaxWhole( S, x)
struct SEMANTIC_NETWORK *S;
int x;  /* the node */
{
	int y, max;

/*  printf("ENTER: LeastMaxWhole\n");*/
	max = x;
	y = x;

	while( y != NO_WHOLE) {
		y = WholeOf( S, y);
		if (y != NO_WHOLE) {
			if (Index[y] > Index[max]) {
/*              printf("Index[<%s>] > Index[<%s>]\n",
					GetName( S, y), GetName( S, max));*/
				max = y;
				}
			}
		}
/* printf("LeastMaxWhole(%s) is %s\n",
				GetName( S, x), GetName( S, max));*/

/* printf("EXIT: LeastMaxWhole\n");*/
   return( max);
}

/* find LMW(F) = { LeastMaxWhole(x) | x in NounsInField(F)} */
cons_ptr AllLeastMaxWholes( S, F, LMW)
struct SEMANTIC_NETWORK *S;
cons_ptr F;
cons_ptr LMW;
{
	cons_ptr NF, nfptr;
	int x;

/*  printf("ENTER: AllLeastMaxWholes\n");*/

	/* find the nouns in F */
	NF = NounsInField( S, F);
	nfptr = NF;
	while (not( null( nfptr))) {
		x = car( nfptr);
		LMW = list_union( LMW, ncons( LeastMaxWhole( S, x)));
		nfptr = cdr( nfptr);
		}
/*  printf("EXIT: AllLeastMaxWholes\n");*/
	return( LMW);
}

/* climb up the mereological chain of x and increment the Index */
/* of each whole, since that whole contains x */
void DecorateMereologicalChainOf(S, x)
struct SEMANTIC_NETWORK *S;
int x;
{
	int y;

/*  printf("ENTER: DecorateMereologicalChainOf\n");
	printf("Decorate chain of <%s>\n", GetName( S, x));*/

	y = x; /* mereological chain begins with x */
	while( y != NO_WHOLE ) {
		Index[y] += 1;
/*      printf("WholeOf(%s) ", GetName( S, y));*/
		y = WholeOf(S,y); /* ascend mereological hierarchy */
/*      printf("is %s\n",(y==NO_WHOLE)?"NIL":GetName( S, y));*/
		}
/*  printf("EXIT: DecorateMereologicalChainOf\n");*/
}

/* for each noun in the field F, increment the Index of each whole in */
/* that noun's mereological chain.  Thus the Index of each whole gets */
/* set as a side-effect to the total number of nouns in F that are */
/* contained by that whole. */
void DecorateAllTermsIn( S, F)
struct SEMANTIC_NETWORK *S;
cons_ptr F; /* the semantic field */
{
	cons_ptr NF, nfptr;
	int x;

/*  printf("ENTER: DecorateAllTermsIn\n");*/
	NF = NounsInField( S, F);

	nfptr = NF;
	while( not( null( nfptr))) {
		x = car( nfptr);
		DecorateMereologicalChainOf(S, x);
		nfptr = cdr(nfptr);
		}
/*  printf("EXIT: DecorateAllTermsIn\n");*/
}


/* compute the mereological closure of a field F using the method */
/* of least maximal wholes.  Returns the */
/* closure of the field as a list of terms */
cons_ptr LeastMaxWholesClosure( S, F)
struct SEMANTIC_NETWORK *S;
cons_ptr F;
{
	int i, x;
	cons_ptr LMW, lmwptr, parts, TheParts;

/*  printf("ENTER: MereologicalClosure\n");*/
	/* initialize Index to zero */
	for (i=0; i < MAX_NODES; i++) Index[i] = 0;

	/* compute the value of Index for each noun in F */
	DecorateAllTermsIn(S, F);

	/* find least maximal containing wholes for nouns in F */
	LMW = NIL;
	LMW = AllLeastMaxWholes( S, F, LMW);

	/* compute the mereological closure of F */
	/* NOTICE that x must be consed onto AllPartsOf(x) */
	/*      since x is not a part of x and isn't in AllPartsOf(x) */
	/* for every least maximal whole, add all its parts to the field */
	lmwptr = LMW;
	while( not( null( lmwptr))) {
		x = car( lmwptr);
/*      printf("LeastMaximalWhole is: <%s>", GetName( S, x));*/
		/* x and all its parts are added to the field */
		TheParts = NIL;
		TheParts = AllPartsOf(S, x, TheParts);
		TheParts = cons( x, TheParts);
/*      printf("Parts added to field: ");
		print_node_names( TheParts, S);*/
		F = list_union( F, TheParts);
		lmwptr = cdr( lmwptr);
		}
/*  printf("EXIT: Mereological Closure\n");*/
	return( F);
}

/***********************************************************************/
/*                                                                     */
/*             Mereological Closure by method of maximal wholes        */
/*                                                                     */
/***********************************************************************/

/* climb up the mereological chain of x to find the */
/* maximal containing whole of x */
int MaxWhole( S, x)
struct SEMANTIC_NETWORK *S;
int x;  /* the node */
{
	int y;

/*  printf("ENTER: MaxWhole\n");*/

	/* climb to the top of the mereological chain of x */
	y = x;  /* y climbs the chain */
	while( WholeOf( S, y) != NO_WHOLE) {
		y = WholeOf( S, y);
		}
/*      printf("MaxWhole(%s) is %s\n",
		GetName( S, x), GetName( S, y));*/

/*  printf("EXIT: MaxWhole\n");*/
	return( y);
}

/* find MW(F) = { MaxWhole(x) | x in NounsInField(F)} */
cons_ptr AllMaxWholes( S, F)
struct SEMANTIC_NETWORK *S;
cons_ptr F;
{
	cons_ptr MW;
	cons_ptr NF, nfptr;
	int x;

/*  printf("ENTER: AllMaxWholes\n");*/

	MW = NIL;

	/* find the nouns in F */
	NF = NounsInField( S, F);
	nfptr = NF;
	while (not( null( nfptr))) {
		x = car( nfptr);
		MW = list_union( MW, ncons( MaxWhole( S, x)));
		nfptr = cdr( nfptr);
		}
/*  printf("EXIT: AllMaxWholes\n");*/
	return( MW);
}


/* compute the mereological closure of a field F using the method */
/* of maximal wholes.  Returns the closure of the field as a list */
/* of terms */
cons_ptr MaxWholesClosure( S, F)
struct SEMANTIC_NETWORK *S;
cons_ptr F;
{
	int x;
	cons_ptr MW, mwptr, parts, TheParts;

/*      printf("ENTER: MaxWholesClosure\n");*/

	/* find maximal containing wholes for nouns in F */
	MW = AllMaxWholes( S, F);

	/* compute the mereological closure of F */
	/* NOTICE that x must be consed onto AllPartsOf(x) */
	/*      since x is not a part of x and isn't in AllPartsOf(x) */
	/* for every maximal whole, add all its parts to the field */
	mwptr = MW;
	while( not( null( mwptr))) {
		x = car( mwptr);
		/* x and all its parts are added to the field */
		TheParts = NIL;
		TheParts = AllPartsOf(S, x, TheParts);
		TheParts = cons( x, TheParts);
/*              printf("Parts added to field: ");*/
/*              print_node_names( TheParts, S);*/
		F = list_union( F, TheParts);
		mwptr = cdr( mwptr);
		}
/*      printf("EXIT: MaxWholesClosure\n");*/
	return( F);
}


/***********************************************************************/
/*                                                                     */
/*                            Mereological Closure                     */
/*                                                                     */
/***********************************************************************/
/* compute the mereological closure of a field F */
cons_ptr MereologicalClosure( S, F, ClosureMethod)
struct SEMANTIC_NETWORK *S;
cons_ptr F;
int ClosureMethod;
{
/*      printf("ENTER: MereologicalClosure\n");*/
	/* choose either method of least maximal wholes (DEFAULT) */
	/* or method of maximal wholes (requires setting a parameter) */
	if (ClosureMethod == LEAST_MAX_WHOLES_METHOD)
		return( LeastMaxWholesClosure( S, F));
	else if (ClosureMethod == MAX_WHOLES_METHOD)
		return( MaxWholesClosure( S, F));
	else {  printf("ERROR: Unrecognized value for ClosureMethod.\n");
		exit(ERROR_VALUE);
		}
/*      printf("EXIT: MereologicalClosure\n");*/
}

/***********************************************************************/
/*              Mereologically Closed Functional Fields                */
/*                                                                     */
/* computes the functional fields of a term x, then computes the       */
/* mereological closure of each functional field and returns the       */
/* closed field.                                                       */
/***********************************************************************/
sequence_ptr closed_func_fields( S, x, ClosureMethod, result)
struct SEMANTIC_NETWORK *S;
int x;
int ClosureMethod;
sequence_ptr result;
{
	sequence_ptr the_func_fields, fptr;
	sequence_ptr closed_fields;
	cons_ptr this_field;

/*  printf("ENTER: closed_func_fields\n");*/
	/* find the semantic field of the node */
	the_func_fields = NIL;
	the_func_fields = func_fields( S, x, the_func_fields);

	/* for each field F in the_func_fields, mereologically close F */
	closed_fields = NIL;
	fptr = the_func_fields;
	while( not (nullseq( fptr))) {
		this_field = carseq( fptr);
		this_field = MereologicalClosure(S, this_field, ClosureMethod);
		this_field = ParadigmaticClosure(S, this_field);
		closed_fields = conseq( this_field, closed_fields);
		fptr = cdrseq( fptr);
		}
/*  printf("EXIT: closed_func_fields\n");*/

	return( closed_fields);
}



/***************************************************************************/
/*                           Structural Fields                             */
/***************************************************************************/

sequence_ptr GrowStructFieldFromNouns( S, FieldNouns, Field)
struct SEMANTIC_NETWORK *S;
cons_ptr FieldNouns;
sequence_ptr Field;
{
	cons_ptr FIDS;          /* FrameIdentifiers in Field */
	cons_ptr TheRels;       /* Relations or Verbs in Field */

	FIDS = NIL;
	FIDS = gen_state_trans( FieldNouns, S, LITERAL, FIDS);
	if (null( FIDS)){
		printf("WARNING: No Frame Identifiers in Field.\n");
		}
/*  printf("FrameIds in field:\n");
	print_node_names( FIDS, S);*/
	TheRels = NIL;
	TheRels = state_trans( FIDS, S, VERB, LITERAL, TheRels);
	if (null(TheRels)) {
		printf("WARNING: No relations or verbs in Field.\n");
		}
/*  printf("Verbs or relations in field:\n");
	print_node_names( TheRels, S);*/
	Field = nconseq( list_union(FieldNouns, list_union(FIDS, TheRels)));
	return( Field);
}

sequence_ptr struct_field( S, cue, Field)
struct SEMANTIC_NETWORK *S;
int cue;
sequence_ptr Field;
{
	int Max;                /* the maximal whole of the cue */
	cons_ptr FieldParts;    /* all parts of the maximal whole */
	cons_ptr FIDS;          /* frame identifiers in the field */
	cons_ptr TheRels;       /* relations in the field */

/*  printf("ENTER: struct_field\n");*/
/*  printf("Find structural field of <%s>\n", GetName( S, cue));*/
	Max = MaximalWholeOf( S, cue);
/*  printf("MaximalWholeOf(<%s>) = <%s>\n",
		GetName( S, cue), GetName( S, Max));*/
	FieldParts = NIL;
	FieldParts = AllPartsOf( S, Max, FieldParts);
	FieldParts = cons( Max, FieldParts);
/*  printf("AllPartsOf(<%s>):\n", GetName( S, Max));
	print_node_names( FieldParts, S);*/
	Field = NIL;
	Field = GrowStructFieldFromNouns( S, FieldParts, Field);
/*  strike_any_key();*/  /* pause while debugging */
/*  printf("EXIT: struct_field\n");*/
	return( Field);
}

/**************************************************************************/
/*                                                                        */
/*                           Generate Fields                              */
/*                                                                        */
/**************************************************************************/

/* select the type of field(s) to be generated from the cue */
int fields_type_menu(S, cue)
struct SEMANTIC_NETWORK *S;
int cue;        /* the cue from which the field is being generated */
{
	char ExtraMsgString[80];

	DisplayWindow( ErrorWindow);
	sprintf(
		ExtraMsgString,
		"Select the type of field(s) for <%s>: ",
		GetName( S, cue));
	DisplayWindow( FieldWindow);
	DisplayExtraMsg( FieldWindow, ExtraMsgString);
	return( get_option( FieldWindow));
}


/* Given a semantic network S and a cue x, generate the field(s) */
/* of x by the selected method */
sequence_ptr get_fields( S, x, fields_type, FieldsOfCue)
struct SEMANTIC_NETWORK *S;
int x;
int fields_type;
sequence_ptr FieldsOfCue;
{
/*  printf("ENTER: get_fields\n"); */
	switch (fields_type) {
	   case 1: /* fields are functional w/ no mereological closure */
			FieldsOfCue = func_fields( S, x, FieldsOfCue);
			break;
	   case 2: /* fields are functional w/ mereological closure */
			/* by method of least maximal wholes */
			FieldsOfCue = closed_func_fields( S, x,
							LEAST_MAX_WHOLES_METHOD, FieldsOfCue);
			break;
	   case 3: /* fields are functional w/ mereological closure */
			/* by method of maximal whole */
			FieldsOfCue = closed_func_fields( S, x,
								MAX_WHOLES_METHOD, FieldsOfCue);
			break;
	   case 4: /* field is structural */
			FieldsOfCue = struct_field( S, x, FieldsOfCue);
			break;
	   default:
			printf("ERROR: Unexpected field type: %d\n", fields_type);
			exit( ERROR_VALUE);
	   }

/*  printf("EXIT: get_fields\n");*/
	return( FieldsOfCue);
}


/* Select a method for finding the fields of a cue */
/* Find the fields of a cue by the selected method */
sequence_ptr fields( S, x, FieldsOfCue)
struct SEMANTIC_NETWORK *S;
int x;
sequence_ptr FieldsOfCue;
{
	int fields_type;  /* the method for generating the fields */

/*  printf("ENTER: fields\n");*/
	fields_type = fields_type_menu( S, x);
	FieldsOfCue = get_fields( S, x, fields_type, FieldsOfCue);
/*  printf("EXIT: fields\n"); */

	return( FieldsOfCue);
}


/**************************************************************************/
/*                                                                        */
/*                          FIELD PRINTING FUNCTIONS                      */
/*                                                                        */
/**************************************************************************/

/* Print the mereological relations in a field as follows: */
/* for each node N in FIELD do {                           */
/*       for each part P of N do {                         */
/*            if P is in FIELD                             */
/*            then printf("contains( <N>, <P>)");}}        */
print_field_parts( S, this_field)
struct SEMANTIC_NETWORK *S;
cons_ptr this_field;
{
	cons_ptr p, q, the_parts;
	int node;

	q = this_field;
	while( not( null( q))) {
		node = car( q);
		the_parts = NIL;
		the_parts = node_trans( node,
			 S, WHOLE, LITERAL, the_parts);
		p = the_parts;
		while( not( null( p))) {
		    if (is_member(car(p), this_field)) {
			  printf("     ");  /* a little indentation */
			  printf("STRUCTURE:contains(WHOLE: %s , PART: %s)\n",
				 GetName( S, node), GetName( S, car(p)));
			  }
		    p = cdr(p); /* get next p */
		    }
		 q = cdr(q);
		 }
}



/**************************************************************************/
/*                                                                        */
/*                     FIELD BUFFER PRINTING FUNCTIONS                    */
/*                                                                        */
/**************************************************************************/
FieldPartsToBuffer( S, FB, this_field)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
cons_ptr this_field;
{
	cons_ptr p, q, the_parts;
	int node;

	ListIterator( q, this_field) {
		node = car( q);
		the_parts = NIL;
		the_parts = node_trans( node, S, WHOLE, LITERAL, the_parts);
		ListIterator( p, the_parts) {
		    if (is_member(car(p), this_field)) {
			  sprintf(CurrentLine(FB),
				 "%s contains %s",
				 GetName( S, node), GetName( S, car(p)));
			  NextLine( FB);
			  }
		    }
		 }
}

void FillFieldBuffer( S, FB, the_fields)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
struct TextBuffer *FB;
sequence_ptr the_fields;        /* the set of fields */
{
	sequence_ptr t;
	int field_num, node;
	cons_ptr ptr, this_field, the_parts;

	field_num = 1;
	SequenceIterator( t, the_fields) {
		sprintf(CurrentLine(FB), "Field %d", field_num);
		NextLine( FB);
		this_field = carseq( t);

		ListIterator( ptr, this_field) {
			node = car( ptr);
			switch (GetType(S, node)) {
				case SFRAME:        SynFrameToBuffer(S, FB, node);
									break;
				case PFRAME:        ParaFrameToBuffer( S, FB, node);
									break;
				case RULE:          RuleToBuffer( S, FB, node);
									break;
				case PROPERTYTYPE:  PropertyToBuffer( S, FB, node);
									break;
				case NOUN:          PrintFeaturesToBuffer( S, FB, node);
									break;
				}
			} /* end iterator over nodes in this_field */
		/* strike_any_key(); */  /* pause for debugging */

		/* print all the mereological frames in the field */
		FieldPartsToBuffer( S, FB, this_field);

		field_num++; /* go on to next field */
		NextLine( FB);
		NextLine( FB);
		} /* end while not nullseq t */
}


/* ask the user for a node and determine its semantic field */
/* This function is called from the main menu */
cons_ptr find_fields( S)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
{
	int index, i;
	int still_doing_fields;
	sequence_ptr the_fields;
	struct TextBuffer *FB;

	still_doing_fields = TRUE;
	while (still_doing_fields) {
		DisplayTermsInNetwork( S, TermsWindow);
		DisplayWindow( FieldNodeQueryWindow);
		DisplayWindow( ErrorWindow);
		/* get a node from the user */
		index = query_user_for_node(S, FieldNodeQueryWindow);
		_clearscreen( _GCLEARSCREEN);

		/* find the semantic fields of the node */
		the_fields = NIL;
		the_fields = fields( S, index, the_fields);

		/* fill the FieldBuffer with the fields */
		FB = AllocateTextBuffer();
		InitializeTextBuffer( FB);
		StringMove( FB, "-------- Please Scroll Down -------");
		NextLine( FB);
		sprintf( CurrentLine(FB),
				"The semantic fields of <%s> are:",
				GetName( S, index));
		NextLine( FB);
		FillFieldBuffer( S, FB, the_fields);
		sprintf(CurrentLine(FB), "***** End of Text *****");
		NextLine( FB);
		DisplayBuffer( FB, FieldScrollWindow);

		/* continue generating fields for terms? */
		DisplayWindow( FieldChoiceWindow);
		QueryAndResponse( FieldChoiceWindow,
						  "Continue generating fields? (Y or N):    [  ]");
		if (GetYesOrNoFromWindow( FieldChoiceWindow) == NO)
			still_doing_fields = FALSE;
		}
		free( FB);
}


/* generate one and only one field from a cue */
cons_ptr find_one_field( S, cue, result)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
int cue;                        /* the cue to generate the field */
cons_ptr result;                /* the resultant field */
{
	sequence_ptr field_seq;

/*  printf("Within find_one_field() the cue is: %s\n",
	GetName( S, cue)); */
	/* generate the fields in which the cue sits */
	field_seq = NIL;
	field_seq = fields(S, cue, field_seq);

	/* if the cue sits in more than one field, */
	/* one must be selected. */
	/* next version will ask the user */
	result = carseq( field_seq);

	return( result);
}

