/***********************************************************************/
/*                                                                     */
/*                                  FRAMES                             */
/*                Copyright (C) 1992 Eric Charles Steinhart            */
/*                            All rights reserved.                     */
/*                                                                     */
/***********************************************************************/
#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\string.h"
#include "anadefs.h"
#include "semnet.h"
#include "buffer.h"
#include "iterate.h"

#define PrintOpenBracketIf( M)  if (M) StringMove( FB,"[")
#define PrintCloseBracketIf( M)  if (M) StringMove( FB,"]")

/***********************************************************************/
/*                    Printing the Terms in the Network                */
/***********************************************************************/
void DisplayTermsInNetwork( S, Window)
struct SEMANTIC_NETWORK *S;
struct WindowStruct *Window;
{
	int node, TimesInCol, row, col;
	char NodeName[MAX_STRLEN];

	_clearscreen( _GCLEARSCREEN);
	DisplayWindow( Window);

	row = 3;
	TimesInCol = 0;

	IntegerIterator( node, 0, GetNumNodes( S)) {
		switch (GetType(S,node)) {
			case NOUN:
			case VERBTYPE:
			case ADJECTIVE:
			case ADVERB:
				col = (TimesInCol*15)+12;
				sprintf(NodeName, "%s\n", GetName(S,node));
				posprint(row, col, NodeName);
				TimesInCol++;
				if (TimesInCol > 3) {
					TimesInCol = 0;
					row++;
					}
				break;
			}
		}
}

void PrintTermsInNetwork( S)
struct SEMANTIC_NETWORK *S;
{
	DisplayTermsInNetwork( S, WordsWindow);
	ExitFromWindow( WordsWindow);
}



/***********************************************************************/
/*         Printing the Frames in the Network to a Line Point          */
/***********************************************************************/
void DescriptionToLinePoint( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;
{
	int position;
	int Modifier, Head, Role;
	int MetaphoricalFrame, MetaphoricalTerm;
	char temp[MAX_STRLEN];

	MetaphoricalFrame = FALSE;
	IntegerIterator( position, 1, MAX_SURFACE_POSITIONS) {
		if (GetModeOfPosition(S, FrameId, position) == ANALOGICAL)
			MetaphoricalFrame = TRUE;
		}
	PrintOpenBracketIf( MetaphoricalFrame);

	StringMove( FB, "the ");

	IntegerIterator( position, 1, MAX_SURFACE_POSITIONS) {
		Head = GetHeadAtPosition(S, FrameId, position);
		if (Head != NO_FILLER) {
			if (position > 1) {
				Role = GetRoleOfPosition(S, FrameId, position);
				if (Role >= FIRST_PREPOSITION) {
					strcpy(temp, role_names[Role]);
					StringMove( FB, strlwr( temp));
					StringMove( FB, " ");
					}
				}
			if (GetModeOfPosition(S, FrameId, position) == ANALOGICAL)
				MetaphoricalTerm = TRUE;
			else MetaphoricalTerm = FALSE;
			PrintOpenBracketIf( MetaphoricalTerm);
			StringMove( FB, GetName(S,Head));
			PrintCloseBracketIf( MetaphoricalTerm);
			StringMove( FB, " ");
			}
		}
	PrintCloseBracketIf( MetaphoricalFrame);
}


void PropertyToLinePoint( S, FB, Head)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int Head;
{
	int Attribute, Object;
	int MetaphoricalAttr;

	Attribute = GetAttributeTerm(S, Head);
	if (GetMode(S, GetConn(S, Attribute, Head)) == ANALOGICAL)
		MetaphoricalAttr = TRUE;
	else MetaphoricalAttr = FALSE;

	PrintOpenBracketIf( MetaphoricalAttr);
	StringMove( FB, GetName(S,Attribute));
	PrintCloseBracketIf( MetaphoricalAttr);

	StringMove( FB, " of ");
	Object = GetObjectTerm(S, Head);
	switch (GetType(S, Object)) {
		case NOUN:
			StringMove( FB, GetName(S,Object));
			break;
		case DESCRIPTION:
			DescriptionToLinePoint(S, FB, Object);
			break;
			}
}

void SynFrameToLinePoint( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;
{
	int position;
	int Modifier, Head, Role;
	int MetaphoricalFrame, MetaphoricalTerm;
	char temp[MAX_STRLEN];
	int Attribute, Object;

	MetaphoricalFrame = FALSE;
	IntegerIterator( position, 1, MAX_SURFACE_POSITIONS) {
		if (GetModeOfPosition(S, FrameId, position) == ANALOGICAL)
			MetaphoricalFrame = TRUE;
		}

	PrintOpenBracketIf( MetaphoricalFrame);
	IntegerIterator( position, 1, MAX_SURFACE_POSITIONS) {
		Modifier = GetModifierAtPosition(S, FrameId, position);
		if (Modifier != NO_FILLER) {
			StringMove(FB, GetName(S,Modifier));
			StringMove(FB, " ");
			}
		Head = GetHeadAtPosition(S, FrameId, position);
		if (Head != NO_FILLER) {

			/* deal with possible prepositions */
			if (position == 3) {
				Role = GetRoleOfPosition(S, FrameId, position);
				if (Role >= FIRST_PREPOSITION) {
					strcpy(temp, role_names[Role]);
					StringMove( FB, strlwr( temp));
					StringMove( FB, " ");
					}
				}
			if (position == 4) {
				/* print relevant preposition for fourth case slot */
				Role =  GetRoleOfPosition(S, FrameId, position);
				switch( Role) {
					case SOURCE:
						StringMove( FB, "from ");
						break;
					case RECIPIENT:
						StringMove( FB, "to ");
						break;
					case INSTRUMENT:
						StringMove( FB, "with ");
						break;
					default:
						strcpy(temp, role_names[Role]);
						StringMove( FB, strlwr( temp));
						StringMove( FB, " ");
					}
				}

			if (GetModeOfPosition(S, FrameId, position) == ANALOGICAL)
				MetaphoricalTerm = TRUE;
			else MetaphoricalTerm = FALSE;
			PrintOpenBracketIf( MetaphoricalTerm);
			switch (GetType(S,Head)) {
				case NOUN:
				case VERBTYPE:
				case SFRAME:
				case MFRAME:
					 StringMove( FB, GetName(S,Head));
					 break;
				case DESCRIPTION:
					 DescriptionToLinePoint( S, FB, Head);
					 break;
				case PROPERTYTYPE:
					 PropertyToLinePoint( S, FB, Head);
					 break;
				default:
					 printf("ERROR: Position <%d> in <%s> has bad type.\n",
							 position, GetName(S,FrameId));
					 exit( ERROR_VALUE);
				}
			PrintCloseBracketIf( MetaphoricalTerm);
			StringMove( FB, " ");
			}
		}
	PrintCloseBracketIf( MetaphoricalFrame);
}


void FeatureToLinePoint( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int FrameId;
{
	cons_ptr Subjects, Predicates;
	int TheSubject, ThePredicate;
	int MetaphoricalFeature;

	MetaphoricalFeature = FALSE;

	/* find the SUBJECT of the FrameId */
	Subjects = NIL;
	Subjects = node_trans(FrameId, S, SUBJECT, LITERAL, Subjects);
	if (Subjects == NIL) {
		/* no LITERAL subjects, look for ANALOGICAL subjects */
		Subjects = node_trans(FrameId, S, SUBJECT, ANALOGICAL, Subjects);
		if (Subjects == NIL) {
			printf("ERROR: Frame <%s> has no subjects.\n",
					GetName(S,FrameId));
			exit( ERROR_VALUE);
			}
		MetaphoricalFeature = TRUE;
		}
	TheSubject = car(Subjects);

	/* find the PREDICATE of the FrameId */
	Predicates = NIL;
	Predicates = node_trans(FrameId, S, PREDICATE, LITERAL, Predicates);
	if (Predicates == NIL) {
		/* no LITERAL predicates, look for ANALOGICAL predicates */
		Predicates = node_trans(FrameId, S, PREDICATE, ANALOGICAL, Predicates);
		if (Predicates == NIL) {
			printf("ERROR: Frame <%s> has no predicates.\n",
					GetName(S,FrameId));
			exit( ERROR_VALUE);
			}
		MetaphoricalFeature = TRUE;
		}
	ThePredicate = car(Predicates);

	PrintOpenBracketIf( MetaphoricalFeature);
	StringMove( FB, GetName(S,TheSubject));
	if (not(MetaphoricalFeature)) StringMove( FB, " is ");
	else StringMove( FB, " [is] ");
	PrintOpenBracketIf( MetaphoricalFeature);
	StringMove( FB, GetName(S,ThePredicate));
	PrintCloseBracketIf( MetaphoricalFeature);
	PrintCloseBracketIf( MetaphoricalFeature);
	FreeList( Subjects);
	FreeList( Predicates);
}


void ConjunctionListToLinePoint( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;                  /* the conjunction node */
{
	cons_ptr LitBranches, AnaBranches, ptr;
	int ThisBranch;

	/* get all the LITERAL ANDBRANCHes of the conjunction node */
	LitBranches = NIL;
	LitBranches = node_trans( FrameId, S, ANDBRANCH, LITERAL, LitBranches);

	/* get all the LITERAL ANDBRANCHes of the conjunction node */
	AnaBranches = NIL;
	AnaBranches = node_trans( FrameId, S, ANDBRANCH, ANALOGICAL, AnaBranches);

	/* print all the ANDBRANCHes to the buffer */
	StringMove( FB, "{ ");
	ListIterator( ptr, LitBranches) {
		ThisBranch = car(ptr);
		StringMove( FB, GetName(S,ThisBranch));
		if (not(null(cdr(ptr)) && null(AnaBranches)))
			StringMove( FB, ", ");
		}
	ListIterator( ptr, AnaBranches) {
		ThisBranch = car(ptr);
		StringMove( FB, "[");
		StringMove( FB, GetName(S,ThisBranch));
		StringMove( FB, "]");
		StringMove( FB, (null(cdr(ptr)))?"":", ");
		}
	StringMove( FB, "}");

	FreeList( LitBranches);
	FreeList( AnaBranches);
}


void FrameToLinePoint( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int FrameId;
{
	switch (GetType(S,FrameId)) {
		case SFRAME:        SynFrameToLinePoint( S, FB, FrameId);
							NextLine(FB);
							break;
		case MFRAME:        FeatureToLinePoint( S, FB, FrameId);
							NextLine(FB);
							break;
		default:            printf("ERROR: Unrecognized frame type.\n");
							exit( ERROR_VALUE);
							break;
		}
}


/***********************************************************************/
/*             Printing the Frames in the Network to a Buffer          */
/***********************************************************************/
void SynFrameToBuffer( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;
{
	StringMove( FB, GetName(S,FrameId));
	StringMove( FB, ": ");
	SynFrameToLinePoint(S, FB, FrameId);
	NextLine(FB);
}


void ParaFrameToBuffer( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;
{
	int PosTerm, NegTerm;

	PosTerm = GetPositiveTerm(S, FrameId);
	if (PosTerm == NO_FILLER) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:No positive term for frame <%s>\n",
			GetName(S,FrameId));
		exit(ERROR_VALUE);
		}

	NegTerm = GetNegativeTerm(S, FrameId);
	if (NegTerm == NO_FILLER) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:No negative term for frame <%s>\n",
			GetName(S,FrameId));
		exit(ERROR_VALUE);
		}

	sprintf(CurrentLine( FB),
		"%s: '%s' is the opposite of '%s'",
		GetName(S,FrameId),
		GetName(S,PosTerm),
		GetName(S,NegTerm));
	NextLine(FB);
}

void RuleToBuffer( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;
{
	int Antecedent, Consequent;

	Antecedent = GetAntecedent(S, FrameId);
	if (Antecedent == NO_FILLER) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:No antecedent for rule <%s>\n",
			GetName(S,FrameId));
		exit(ERROR_VALUE);
		}

	Consequent = GetConsequent(S, FrameId);
	if (Consequent == NO_FILLER) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:No consequent for rule <%s>\n",
			GetName(S,FrameId));
		exit(ERROR_VALUE);
		}

	StringMove( FB, GetName(S, FrameId));
	StringMove( FB, ": if ");
	StringMove( FB, GetName(S, Antecedent));
	StringMove( FB, " then ");
	if (GetType(S, Consequent) == CONJUNCTION)
		ConjunctionListToLinePoint(S, FB, Consequent);
	else {
		StringMove( FB, "{ ");
		StringMove( FB, GetName(S,Consequent));
		StringMove( FB, "}");
		}
	NextLine(FB);
}

void PropertyToBuffer( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;
{
	int Attribute, Object, ThisValue, Connection;
	cons_ptr Values, AnaValues, ptr;

	Attribute = GetAttributeTerm(S, FrameId);
	if (Attribute == NO_FILLER) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:No attribute for property <%s>\n", GetName(S,FrameId));
		exit(ERROR_VALUE);
		}
	Connection = GetConn(S, FrameId, Attribute);

	StringMove( FB, GetName(S,FrameId));
	StringMove( FB, ": ");
	if (GetMode( S,Connection) == ANALOGICAL) StringMove( FB, "[");

	Object = GetObjectTerm(S, FrameId);
	if (Object == NO_FILLER) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:No object for property <%s>\n", GetName(S,FrameId));
		exit(ERROR_VALUE);
		}

	StringMove( FB, "The ");
	if (GetMode( S,Connection) == ANALOGICAL) StringMove( FB, "[");
	StringMove( FB, GetName(S,Attribute));
	if (GetMode( S,Connection) == ANALOGICAL) StringMove( FB, "]");
	StringMove( FB, " of ");
	switch (GetType(S, Object)) {
		case NOUN:
			 StringMove( FB, GetName(S,Object));
			 break;
		case DESCRIPTION:
			 DescriptionToLinePoint(S, FB, Object);
			 break;
		}
	StringMove( FB, " is one of { ");

	Values = NIL;
	Values = GetLiteralValues(S, FrameId, Values);
	AnaValues = NIL;
	AnaValues = GetAnalogicalValues(S, FrameId, AnaValues);

	if (null(Values) && null(AnaValues)) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:No values for property <%s>\n",
			GetName(S,FrameId));
		exit(ERROR_VALUE);
		}

	ListIterator( ptr, Values) {
		ThisValue = car(ptr);
		StringMove( FB, GetName(S,ThisValue));
		StringMove( FB, (null(cdr(ptr)) && null(AnaValues))?"":", ");
		}

	ListIterator( ptr, AnaValues) {
		ThisValue = car(ptr);
		StringMove( FB, "[");
		StringMove( FB, GetName(S,ThisValue));
		StringMove( FB, "]");
		StringMove( FB, (null(cdr(ptr)))?"":", ");
		}
	StringMove( FB, "}");

	if (GetMode( S,Connection) == ANALOGICAL)
		StringMove( FB, "]");
	NextLine(FB);

	FreeList( Values);
	FreeList( AnaValues);
}

void PrintSupertypesToBuffer( S, FB, the_node)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int the_node;
{
	cons_ptr t, the_supertypes;

	/* the_supertypes = { X | the_node is a SUBTYPE of X} */
	/* hence the_node is a kind of X */
	the_supertypes = NIL;
	the_supertypes = node_trans( the_node, S,
									SUBTYPE, LITERAL, the_supertypes);
	ListIterator( t, the_supertypes) {
		sprintf(CurrentLine( FB),
			"%s is a kind of %s",
			GetName(S,the_node), GetName(S, car(t)));
		NextLine(FB);
		}
	FreeList( the_supertypes);
}


void PrintPartsToBuffer( S, FB, the_node)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int the_node;
{
	cons_ptr t, the_parts;

	/* print any MEREOLOGICAL frames in which the node is a WHOLE */
	the_parts = NIL;
	the_parts = node_trans( the_node, S, WHOLE, LITERAL, the_parts);
	ListIterator( t, the_parts) {
		sprintf(CurrentLine( FB),
			"%s contains %s",
			GetName(S,the_node), GetName(S, car(t)));
		NextLine(FB);
		}
	FreeList( the_parts);
}

void PrintWholesToBuffer( S, FB, the_node)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int the_node;
{
	cons_ptr t, the_wholes;

	/* print any MEREOLOGICAL frames in which the node is a PART */
	the_wholes = NIL;
	the_wholes = node_trans( the_node, S, PART, LITERAL, the_wholes);
	ListIterator( t, the_wholes) {
		sprintf(CurrentLine( FB),
			"%s contains %s",
			GetName(S, car(t)), GetName(S,the_node));
		NextLine(FB);
		}
	FreeList( the_wholes);
}

void PrintFeaturesToBuffer( S, FB, the_node)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int the_node;
{
	cons_ptr t, s, FeatureFrames, ThePredicates;

	/* find the feature frames in which the node is a LITERAL SUBJECT */
	FeatureFrames = NIL;
	FeatureFrames = node_trans(the_node, S, SUBJECT, LITERAL, FeatureFrames);
	ListIterator( t, FeatureFrames) {
		/* find the predicate of each feature frame */
		ThePredicates = NIL;
		ThePredicates = node_trans(car(t), S, PREDICATE,
										LITERAL, ThePredicates);
		ListIterator( s, ThePredicates) {
			sprintf(CurrentLine( FB),
				"%s: %s is %s",
				GetName(S,car(t)),
				GetName(S,the_node),
				GetName(S, car(s)));
			NextLine(FB);
			}
		}

	/* print FEATURE frames in which the node is a ANALOGICAL SUBJECT */
	FeatureFrames = NIL;
	FeatureFrames = node_trans(the_node, S, SUBJECT,
									ANALOGICAL, FeatureFrames);
	ListIterator( t, FeatureFrames) {
		/* find the predicate of each feature frame */
		ThePredicates = NIL;
		ThePredicates = node_trans(car(t), S, PREDICATE,
										ANALOGICAL, ThePredicates);
		ListIterator( s, ThePredicates) {
			sprintf(CurrentLine( FB),
				"%s: %s [is] [%s]",
				GetName(S,car(t)),
				GetName(S,the_node),
				GetName(S, car(s)));
			NextLine(FB);
			}
		}
	FreeList( FeatureFrames);
	FreeList( ThePredicates);
}

void DescriptionToBuffer( S, FB, FrameId)
struct SEMANTIC_NETWORK *S;   /* the semantic network */
struct TextBuffer *FB;
int FrameId;
{
	DescriptionToLinePoint(S, FB, FrameId);
	NextLine(FB);
}

void PrintNodesFramesToBuffer( S, FB, i)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int i;
{
		switch (GetType(S,i)) {
			case SFRAME:
				SynFrameToBuffer( S, FB, i);
				break;
			case PFRAME:
				ParaFrameToBuffer( S, FB, i);
				break;
			case RULE:
				RuleToBuffer( S, FB, i);
				break;
/*          NO LONGER PRINTING CONJUNCTIONS */
/*          case CONJUNCTION:
				ConjunctionToBuffer( S, FB, i);
				break;*/
			case PROPERTYTYPE:
				PropertyToBuffer( S, FB, i);
				break;
/*          case DESCRIPTION:
				DescriptionToBuffer( S, FB, i);
				break;*/
			case NOUN:
				PrintPartsToBuffer( S, FB, i);
				PrintSupertypesToBuffer( S, FB, i);
				PrintFeaturesToBuffer( S, FB, i);
				break;
			}
}

void PrintListOfFrames( S, ListOfFrames, Window)
struct SEMANTIC_NETWORK *S;
cons_ptr ListOfFrames;
struct WindowStruct *Window;
{
	int i;
	int this_node;
	struct TextBuffer *FB;
	cons_ptr ptr;

	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);

	/* fill the buffer with the frames in the network */
	ListIterator( ptr, ListOfFrames) {
		PrintNodesFramesToBuffer( S, FB, car(ptr));
		}

	BackUpOneLine( FB);

	DisplayBuffer( FB, Window);
	free( FB);
}


void PrintNetworkFramesToBuffer( S)
struct SEMANTIC_NETWORK *S;
{
	int i, j;
	int this_node;
	struct TextBuffer *FB;

	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);
	DisplayWindow( FrameScrollWindow);
	DisplayWindow( FieldChoiceWindow);
	DisplayExtraMsg( FieldChoiceWindow, "Compiling propositions....    ");

	/* fill the buffer with the frames in the network */
	IntegerIterator( i, 0, GetNumNodes( S)) {
		PrintNodesFramesToBuffer( S, FB, i);
		}

	BackUpOneLine( FB);

	DisplayBuffer( FB, FrameScrollWindow);
	free( FB);
}



/* ask the user for a node and determine its frames */
void find_frames( S)
struct SEMANTIC_NETWORK *S;     /* the semantic network */
{
	int the_node, i;
	cons_ptr t, the_frames, ParaFrames, the_wholes, the_parts;
	struct TextBuffer *FB;

	DisplayTermsInNetwork( S, TermsWindow);
	DisplayWindow( FrameNodeQueryWindow);
	DisplayWindow( ErrorWindow);
	/* get a node from the user */
	the_node = query_user_for_node(S, FrameNodeQueryWindow);
	_clearscreen( _GCLEARSCREEN);

	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);

	/* print the SYNTAGMATIC frames in which the_node sits */
	the_frames = NIL;
	the_frames = gen_node_trans( the_node, S, LITERAL, the_frames);
	ListIterator( t, the_frames) {
		if (GetType(S,car(t)) == SFRAME)
				SynFrameToBuffer( S, FB, car(t));
		}

	/* print the PARADIGMATIC frames in which the_node sits */
	ParaFrames = NIL;
	ParaFrames = para_node_trans( the_node, S, LITERAL, ParaFrames);
	ListIterator( t, ParaFrames) {
		if (GetType(S,car(t)) == PFRAME)
			ParaFrameToBuffer( S, FB, car(t));
		}

	if (GetType(S,the_node) == NOUN) {
		PrintWholesToBuffer( S, FB, the_node);
		PrintPartsToBuffer( S, FB, the_node);
		PrintFeaturesToBuffer( S, FB, the_node);
		}

	BackUpOneLine( FB);

	DisplayBuffer( FB, FrameScrollWindow);
	free( FB);
}

