/**************************************************************************/
/*                                                                        */
/*                            Generating Metaphors                        */
/*                  Copyright (C) 1992 Eric Charles Steinhart             */
/*                              All rights reserved.                      */
/**************************************************************************/

#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\string.h"
#include "anadefs.h"
#include "semnet.h"
#include "map.h"
#include "buffer.h"
#include "iterate.h"

#define PrintOpenBracket( FB)  StringMove( FB,"[")
#define PrintCloseBracket( FB) StringMove( FB,"]")

/***************************************************************************/
/*                                                                         */
/*                           NOUN IDENTITY METAPHORS                       */
/*                                                                         */
/***************************************************************************/
sequence_ptr MakePairsOfOpposites(
				S, TopicNoun, SourceNoun, PairsOfOpposites)
struct SEMANTIC_NETWORK *S;
int TopicNoun;
int SourceNoun;
sequence_ptr PairsOfOpposites;
{
	cons_ptr TopicNounFeatures, t;
	cons_ptr SourceNounFeatures, s;

	TopicNounFeatures = NIL;
	TopicNounFeatures = GetFeatures( S, TopicNoun,
							LITERAL, TopicNounFeatures);
/*  printf("TopicNoun <%s> has features: ", GetName(S,TopicNoun));
	print_node_names( TopicNounFeatures, S); printf("\n");*/

	SourceNounFeatures = NIL;
	SourceNounFeatures = GetFeatures( S, SourceNoun,
							LITERAL, SourceNounFeatures);
/*  printf("SourceNoun <%s> has features: ", GetName(S,SourceNoun));
	print_node_names( SourceNounFeatures, S); printf("\n");*/

	ListIterator( t, TopicNounFeatures) {
		ListIterator( s, SourceNounFeatures) {
			if (Opposites(S, car(t), car(s))) {
/*              printf("Opposites <%s> and <%s>\n",
					GetName(S,car(t)), GetName(S,car(s)));*/
				PairsOfOpposites = unionseq(PairsOfOpposites,
									nconseq(cons(car(t), ncons(car(s)))));
				}
			}
		}
	return( PairsOfOpposites);
}

void noun_identity( S, FB, map, target)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
cons_ptr target;
{

	cons_ptr node;          /* index to target */
	int targ_term_node;     /* target node as integer */
	int NOUN1, NOUN2, TopicAdj;
	sequence_ptr PairsOfOpposites, Pair;
	int PROPERTY1, PROPERTY2;
	int attribute1, attribute2;
	int object1, object2;

	StringMove( FB, "Noun-Identification Metaphors.");
	NextLine( FB);

	/* for each node in target */
	ListIterator( node, target) {
	  targ_term_node = car( node);

	  /************ GENERATE NOUN-IDENTITY METAPHORS **********/
	  if ((GetType(S,targ_term_node) == NOUN) &&
			(not(is_attribute(S, targ_term_node)))) {
		 NOUN1 = targ_term_node;
		 NOUN2 = get_analogue( S, map, NOUN1);
		 if (NOUN2 != NIL) {
			sprintf( CurrentLine(FB), "     %s [%s] [%s %s]",
				GetName(S,NOUN1),
				(is_plural(GetName(S,NOUN1)))?"are":"is",
				(is_vowel(GetName(S,NOUN2)))?"an":"a",
				GetName(S,NOUN2));
			NextLine( FB);
			}

		 /******* GENERATE INCONGRUOUS ADJECTIVE METAPHORS *****/
		 PairsOfOpposites = NIL;
		 PairsOfOpposites = MakePairsOfOpposites(
								S, NOUN1, NOUN2, PairsOfOpposites);

		 SequenceIterator( Pair, PairsOfOpposites) {
			TopicAdj = first( carseq( Pair));
			sprintf( CurrentLine(FB), "     %s [%s] [%s %s [%s]]",
				GetName(S,NOUN1),
				(is_plural(GetName(S,NOUN1)))?"are":"is",
				(is_vowel(GetName(S,TopicAdj)))?"an":"a",
				GetName(S,TopicAdj),
				GetName(S,NOUN2));
			NextLine( FB);
			}
		 } /* end if is a noun */

	  /************ GENERATE PROPERTY-IDENTITY METAPHORS **********/
	  if (GetType(S,targ_term_node) == PROPERTYTYPE) {
		 PROPERTY1 = targ_term_node;
		 PROPERTY2 = get_analogue( S, map, PROPERTY1);
		 if (PROPERTY2 != NIL) {
			attribute1 = GetAttributeTerm(S, PROPERTY1);
			object1 = GetObjectTerm(S, PROPERTY1);
			attribute2 = GetAttributeTerm(S, PROPERTY2);
			object2 = GetObjectTerm(S, PROPERTY2);
			sprintf( CurrentLine( FB),
				"     The %s of a %s [is] [the %s of a %s]",
				GetName(S,attribute1),
				GetName(S,object1),
				GetName(S,attribute2),
				GetName(S,object2));
			NextLine( FB);
			}
		 }
	  } /* end for each term in target */
}


/***************************************************************************/
/*                                                                         */
/*              NOUN-IDENTIFICATION METAPHORS WITH ADJECTIVES              */
/*                                                                         */
/***************************************************************************/
void NounIdentityWithAdjectives( S, FB, map, target)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
cons_ptr target;
{

	cons_ptr node;          /* index to target */
	int targ_term_node;     /* target node as integer */
	cons_ptr SourceNounFeatures, s;
	int TargetNoun, SourceNoun, TargetAdj, SourceAdj;

	StringMove( FB, "Noun-Identification Metaphors With Adjectives.");
	NextLine( FB);

	/* for each node in target */
	ListIterator( node, target) {
	  targ_term_node = car( node);
	  /* if targ_term_node is a noun */
	  if (GetType(S,targ_term_node) == NOUN) {
		 TargetNoun = targ_term_node;
		 SourceNoun = get_analogue( S, map, TargetNoun);
		 if (SourceNoun != NIL) {
			SourceNounFeatures = NIL;
			SourceNounFeatures = GetPredicableAdjectives(S,
									SourceNoun, SourceNounFeatures);
			ListIterator( s, SourceNounFeatures) {
				SourceAdj = car(s);
				TargetAdj = get_analogue( S, map, SourceAdj);
				if (TargetAdj != NIL) {
					sprintf( CurrentLine( FB), "     %s %s [%s] [%s %s]",
						GetName(S,TargetAdj),
						GetName(S,TargetNoun),
						(is_plural(GetName(S,TargetNoun)))?"are":"is",
						GetName(S,SourceAdj),
						GetName(S,SourceNoun));
					NextLine( FB);
					}
				}
			}
		 }
	  }
}

/***************************************************************************/
/*                                                                         */
/*                  NOUN IDENTITY WITH GENITIVE METAPHORS                  */
/*                                                                         */
/***************************************************************************/
void mereological_genitive( S, FB, map, target)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
cons_ptr target;
{

	cons_ptr node;          /* index to target */
	int targ_term_node;     /* target node as integer */
	int NOUN1, NOUN2, NOUN3;

	/* for each node in target */
	ListIterator( node, target) {
	  targ_term_node = car( node);
	  /* if targ_term_node is a noun */
	  if (GetType(S,targ_term_node) == NOUN) {
	     NOUN1 = targ_term_node;
	     NOUN2 = get_analogue( S, map, NOUN1);
	     NOUN3 = MaximalWholeOf(S, NOUN1);
		 if (not( NOUN3 == 0) && (NOUN1 != NOUN3)) {
			sprintf(CurrentLine( FB), "     %s [is the %s of] %s",
				GetName(S,NOUN1),
				GetName(S,NOUN2),
				GetName(S,NOUN3));
			NextLine( FB);
			}
	     } /* end if is a term */
	  } /* end for each term in target */
}

/*********************************************************************/
/* Finds noun identity with genitive metaphors                       */
/* using the following algorithm:                                    */
/*                                                                   */
/* for each NOUN1 in TargetNouns do {                                */
/*     NOUN2 = GetAnalogue(NOUN1);                                   */
/*     GenSet = {N in SourceNouns | OF( NOUN2, N)};                  */
/*     for each N in GenSet do {                                     */
/*         NOUN4 = N;                                                */
/*         NOUN3 = GetAnalogue( NOUN4);                              */
/*         print( NOUN1 [is] the [NOUN2] of NOUN3); }}               */
/*                                                                   */
/*********************************************************************/
void functional_genitive( S, FB, map, target)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
cons_ptr target;
{
	cons_ptr node;          /* index to target */
	int targ_term_node;     /* target node as integer */
	cons_ptr GenSet;        /* list of genitives */
	cons_ptr GenSetPtr;     /* pointer to list of genitives */
	int NOUN1, NOUN2, NOUN3, NOUN4, N;

	/* for each node in target */
	ListIterator( node, target) {
	  targ_term_node = car( node);
	  /* if targ_term_node is a noun */
	  if (GetType(S,targ_term_node) == NOUN) {
	     NOUN1 = targ_term_node;
	     NOUN2 = get_analogue( S, map, NOUN1);
		 GenSet = NIL;
	     GenSet = node_trans( NOUN2, S, OF, LITERAL, GenSet);
		 ListIterator( GenSetPtr, GenSet) {
			N = car(GenSetPtr);
			NOUN4 = N;
			NOUN3 = get_analogue(S, map, NOUN4);
			if (not( NOUN3 == 0)) {
			   sprintf(CurrentLine( FB), "     %s [%s the %s of] %s",
				GetName(S,NOUN1),
				(is_plural(GetName(S,NOUN1)))?"are":"is",
				GetName(S,NOUN2),
				GetName(S,NOUN3));
			   NextLine( FB);
			   }
			} /* end GenSetPtr iterator */
	     } /* end if is a term */
	  } /* end for each term in target */
}


void noun_identity_with_genitive( S, FB, map, target)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
cons_ptr target;
{
	StringMove( FB, "Noun-Identification with Genitive Metaphors.");
	NextLine( FB);
	functional_genitive(S, FB, map, target);
	mereological_genitive(S, FB, map, target);
}


/***************************************************************************/
/*                                                                         */
/*                      ADJECTIVE PREDICATION METAPHORS                    */
/*                                                                         */
/***************************************************************************/
void adjective_predication( S, FB, map, target)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
cons_ptr target;
{

	cons_ptr node;          /* index to target */
	int targ_term_node;     /* target node as integer */
	cons_ptr SourceNounFeatures, s;
	int NOUN1, NOUN2, TopicAdj, SourceAdj;

	StringMove( FB, "Adjective-Identification Metaphors.");
	NextLine( FB);

	/* for each node in target */
	ListIterator( node, target) {
	  targ_term_node = car( node);
	  /* if targ_term_node is a noun */
	  if (GetType(S,targ_term_node) == NOUN) {
		 NOUN1 = targ_term_node;
		 NOUN2 = get_analogue( S, map, NOUN1);
		 if (NOUN2 != NIL) {
			SourceNounFeatures = NIL;
			SourceNounFeatures = GetFeatures( S, NOUN2,
									  LITERAL, SourceNounFeatures);
			ListIterator( s, SourceNounFeatures) {
				SourceAdj = car(s);
				TopicAdj = get_analogue( S, map, SourceAdj);
				if (TopicAdj != NIL) {
					sprintf( CurrentLine( FB), "     %s %s [%s] [%s]",
						GetName(S,TopicAdj),
						GetName(S,NOUN1),
						(is_plural(GetName(S,NOUN1)))?"are":"is",
						GetName(S,SourceAdj));
					NextLine( FB);
					}
				}
			}
		 }
	  }
}


/***************************************************************************/
/*                                                                         */
/*                        VERB PREDICATION METAPHORS                       */
/*                                                                         */
/***************************************************************************/
void PrintMetaphoricalDesc( S, FB, map, Description)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
int Description;
{
	int position;
	int Head, Role, AnalogousTargetNoun;
	int MetaphoricalFrame, MetaphoricalTerm;
	char temp[MAX_STRLEN];

	PrintOpenBracket( FB);

	StringMove( FB, "the ");

	IntegerIterator( position, 1, MAX_SURFACE_POSITIONS) {
		Head = GetHeadAtPosition(S, Description, position);
		if (Head != NO_FILLER) {
			if (position > 1) {
				Role = GetRoleOfPosition(S, Description, position);
				if (Role >= FIRST_PREPOSITION) {
					strcpy(temp, role_names[Role]);
					StringMove( FB, strlwr( temp));
					StringMove( FB, " ");
					}
				}
			AnalogousTargetNoun = get_analogue(S, map, Head);
			if (AnalogousTargetNoun != NIL) {
				StringMove( FB, GetName(S,AnalogousTargetNoun));
				StringMove( FB, " ");
				}
			else { /* there is no analogous target noun for this slot */
				PrintOpenBracket( FB);
				StringMove( FB, GetName(S,Head));
				PrintCloseBracket( FB);
				StringMove( FB, " ");
				}
			}
		}
	PrintCloseBracket( FB);
}

void PrintNounPhraseInPosition( S, FB, map, SourceFrame, position)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
int SourceFrame;
int position;
{
	int SrcMod, AnalogousTargetMod;
	int SrcHead, AnalogousTargetNoun;
	int SrcAttr, SrcObject;

	SrcHead = GetHeadAtPosition(S, SourceFrame, position);
	if (SrcHead != NO_FILLER) {
		switch (GetType(S,SrcHead)) {
			case NOUN:
				SrcMod = GetModifierAtPosition(S, SourceFrame, position);
				if (SrcMod != NO_FILLER) {
					AnalogousTargetMod = get_analogue(S, map, SrcMod);
					if (AnalogousTargetMod != NIL) {
						StringMove( FB, GetName(S,AnalogousTargetMod));
						StringMove( FB, " ");
						}
					}
				AnalogousTargetNoun = get_analogue(S, map, SrcHead);
				if (AnalogousTargetNoun != NIL) {
					StringMove( FB, GetName(S,AnalogousTargetNoun));
					StringMove( FB, " ");
					}
				else { /* there is no analogous target noun for this slot */
					PrintOpenBracket( FB);
					StringMove( FB, GetName(S,SrcHead));
					PrintCloseBracket( FB);
					StringMove( FB, " ");
					}
				break; /* end case for source head type is NOUN */
			case PROPERTYTYPE:
				SrcAttr = GetAttributeTerm( S, SrcHead);
				SrcObject = GetObjectTerm( S, SrcHead);
				AnalogousTargetNoun = get_analogue(S, map, SrcObject);
				if (AnalogousTargetNoun != NIL) {
					StringMove( FB, "[[");
					StringMove( FB, GetName(S,SrcAttr));
					StringMove( FB, "] of ");
					StringMove( FB, GetName(S,AnalogousTargetNoun));
					StringMove( FB, "] ");
					}
				else { /* there is no analogous target noun for this slot */
					PrintOpenBracket( FB);
					StringMove( FB, GetName(S,SrcAttr));
					StringMove( FB, " of ");
					StringMove( FB, GetName(S,SrcObject));
					PrintCloseBracket( FB);
					StringMove( FB, " ");
					}
				break; /* end case for source head type is PROPERTYTYPE */
			case DESCRIPTION:
				PrintMetaphoricalDesc( S, FB, map, SrcHead);
				break;
			default:
				printf("ERROR: Position <%d> in <%s> has bad type.\n",
						position, GetName(S,SourceFrame));
				exit( ERROR_VALUE);
			} /* end switch on source head type */
		}
}

void UseGlobalStructLinePoint( S, FB, map, SourceFrame)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
int SourceFrame;
{

	int SrcVerb;
	int Role;
	char Temp[MAX_STRLEN+1];

	/* PROCESS FIRST POSITION: [MODIFIER] NOUN */
	PrintNounPhraseInPosition(S, FB, map, SourceFrame, 1);

	/* PROCESS SECOND POSITION: VERB */
	SrcVerb = GetFrameVerb(S, SourceFrame);
	if (SrcVerb != NO_FILLER) {
		PrintOpenBracket( FB);
		StringMove( FB, GetName(S,SrcVerb));
		PrintCloseBracket( FB);
		StringMove( FB, " ");
		}

	/* PROCESS THIRD POSITION: [PREPOSITION] [MODIFIER] NOUN */
	Role = GetRoleOfPosition(S, SourceFrame, 3);
	if (Role >= FIRST_PREPOSITION) {
		strcpy(Temp, role_names[Role]);
		StringMove( FB, strlwr(Temp));
		StringMove( FB, " ");
		}
	PrintNounPhraseInPosition(S, FB, map, SourceFrame, 3);

	/* PROCESS FOURTH POSITION: [[PREPOSITION] [MODIFIER] NOUN] */
	switch( GetRoleOfPosition(S, SourceFrame, 4)) {
		case SOURCE:     StringMove( FB, "from "); break;
		case RECIPIENT:  StringMove( FB, "to "); break;
		case INSTRUMENT: StringMove( FB, "with "); break;
		}
	PrintNounPhraseInPosition(S, FB, map, SourceFrame, 4);

	NextLine( FB);
}

void UseGlobalStructure( S, FB, map, SourceFrame)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
int SourceFrame;
{
	StringMove( FB, "     ");
	UseGlobalStructLinePoint(S, FB, map, SourceFrame);
}

void PrintTargetAdjective( S, FB, TargetFrame, i)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int TargetFrame;
int i;
{
	int TargetMod;

	TargetMod = GetModifierAtPosition(S, TargetFrame, i);
	if (TargetMod != NO_FILLER) {
		StringMove( FB, GetName(S,TargetMod));
		StringMove( FB, " ");
		}
}


void PrintTargetNoun( S, FB, TargetFrame, i)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
int TargetFrame;
int i;
{
	int TargetNoun;

	TargetNoun = GetNounAtPosition(S, TargetFrame, i);
	if (TargetNoun != NO_FILLER) {
		StringMove( FB, GetName(S,TargetNoun));
		StringMove( FB, " ");
		}
}


void UseLocalStructLinePoint( S, FB, map, SourceFrame)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
int SourceFrame;
{

	int SrcVerb;
	int AnalogousTargetFrame;
	int Role;
	char Temp[MAX_STRLEN];

	AnalogousTargetFrame = get_analogue(S, map, SourceFrame);

	PrintTargetAdjective( S, FB, AnalogousTargetFrame, 1);
	PrintTargetNoun( S, FB, AnalogousTargetFrame, 1);

	SrcVerb = GetFrameVerb(S, SourceFrame);
	if (SrcVerb != NO_FILLER) {
		PrintOpenBracket( FB);
		StringMove( FB, GetName(S,SrcVerb));
		PrintCloseBracket( FB);
		StringMove( FB, " ");
		}

	Role = GetRoleOfPosition(S, SourceFrame, 3);
	if (Role >= FIRST_PREPOSITION) {
		strcpy(Temp, role_names[Role]);
		StringMove( FB, strlwr(Temp));
		StringMove( FB, " ");
		}
	PrintTargetAdjective( S, FB, AnalogousTargetFrame, 3);
	PrintTargetNoun( S, FB, AnalogousTargetFrame, 3);

	/* print relevant preposition for fourth case slot */
	switch( GetRoleOfPosition(S, SourceFrame, 4)) {
		case SOURCE:     StringMove( FB, "from "); break;
		case RECIPIENT:  StringMove( FB, "to "); break;
		case INSTRUMENT: StringMove( FB, "with "); break;
		}

	PrintTargetAdjective( S, FB, AnalogousTargetFrame, 4);
	PrintTargetNoun( S, FB, AnalogousTargetFrame, 4);

	NextLine( FB);
}

void UseLocalStructure( S, FB, map, SourceFrame)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
int SourceFrame;
{
	StringMove( FB, "     ");
	UseLocalStructLinePoint(S, FB, map, SourceFrame);
}

void verb_predication( S, FB, map, source)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
cons_ptr source;
{
	cons_ptr SrcPtr;
	int SourceFrame;
	int AnalogousTargetFrame;

	StringMove( FB, "Verb-Predication Metaphors.");
	NextLine( FB);

	/* for every node in the source */
	ListIterator( SrcPtr, source) {
	  SourceFrame = car( SrcPtr);
	  /* if the node is a syntagmatic frame identifier */
	  if (GetType(S,SourceFrame) == SFRAME) {
		  AnalogousTargetFrame = get_analogue(S, map, SourceFrame);
		  if (AnalogousTargetFrame == NIL)
				UseGlobalStructure(S, FB, map, SourceFrame);
		  else UseLocalStructure(S, FB, map, SourceFrame);
		  }
	  }
}

/* prints source frame with topic nouns */
/* that is, it prints a metaphorical version of the source frame */
void PrintMetaphoricalFrame( S, FB, map, SourceFrame)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct map_structure *map;
int SourceFrame;
{
	int AnalogousTargetFrame;

	AnalogousTargetFrame = get_analogue(S, map, SourceFrame);
	if (AnalogousTargetFrame == NIL)
		UseGlobalStructLinePoint(S, FB, map, SourceFrame);
	else UseLocalStructLinePoint(S, FB, map, SourceFrame);
}

/***************************************************************************/
/*                                                                         */
/*                                 METAPHORS                               */
/*                                                                         */
/***************************************************************************/
void metaphorical_surface_structs( S, map, target, source)
struct SEMANTIC_NETWORK *S;
struct map_structure *map;
cons_ptr target;
cons_ptr source;
{
	int i;
	struct TextBuffer *FB;

	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);

	/* fill the buffer with metaphors */
	StringMove( FB, "Words in brackets [] are used METAPHORICALLY.");
	NextLine( FB);
	NextLine( FB);
	noun_identity( S, FB, map, target);
	NextLine( FB);
	NounIdentityWithAdjectives( S, FB, map, target);
	NextLine( FB);
	noun_identity_with_genitive( S, FB, map, target);
	NextLine( FB);
	adjective_predication( S, FB, map, target);
	NextLine( FB);
	verb_predication( S, FB, map, source);

	BackUpOneLine( FB);
	DisplayBuffer( FB, MetaphorWindow);
	free( FB);
}



/***************************************************************************/
/*                                                                         */
/*                           Generating Metaphors                          */
/*                                                                         */
/***************************************************************************/
/* Performs analogical ACCESS and analogical ELABORATION */
/* exactly like access_and_elaboration(S).               */
/* Generates metaphors after access and elaboration.     */
void generate_metaphors( S)
struct SEMANTIC_NETWORK *S;
{
	int i;
	int target_cue;                 /* the cue */
	int target_type;                /* type of target field */
	int need_target, ThereIsAnError;
	char ErrorMsg[80];
	sequence_ptr cand_targs;        /* candidate target fields */
	cons_ptr target;                /* the target field */
	sequence_ptr cand_srcs;         /* the candidate source fields */
	cons_ptr source;                /* the source field */
	struct map_structure *map;
	struct TextBuffer *FB;

/*  _clearscreen( _GCLEARSCREEN);
	DisplayWindow( StartMetsWindow);
	ExitFromWindow( StartMetsWindow); */

	/*******************************************************************/
	/*            Generate, Display, and Choose Target Field           */
	/*******************************************************************/
	/* generate candidate target fields from the cue */
	need_target = TRUE;
	ThereIsAnError = FALSE;
	while( need_target) {
		_clearscreen( _GCLEARSCREEN);
		DisplayTermsInNetwork( S, TermsWindow);
		DisplayWindow( TopicNodeQueryWindow);
		DisplayWindow( ErrorWindow);
		if (ThereIsAnError) PrintError( ErrorMsg);
		/* get the target cue */
		target_cue = query_user_for_node(S, TopicNodeQueryWindow);
		_clearscreen( _GCLEARSCREEN);
		/* generate target field from the cue */
		cand_targs = NIL;
		target_type = fields_type_menu( S, target_cue);
		cand_targs = get_fields(S, target_cue, target_type, cand_targs);
		if (nullseq( cand_targs)) {
			ThereIsAnError = TRUE;
			sprintf(ErrorMsg, "Target field for <%s> was NIL.",
					GetName(S,target_cue));
			}
		else need_target = FALSE;
		}

	/* fill the FieldBuffer with the candidate target fields */
	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);
	StringMove( FB, "------- Please Scroll Down -------");
	NextLine( FB);
	StringMove( FB, "The target fields are: ");
	NextLine( FB);
	FillFieldBuffer( S, FB, cand_targs);
	StringMove( FB, "***** End of Text *****");
	NextLine( FB);
	DisplayBuffer( FB, FieldScrollWindow);

	/* select a target field for access */
	if (lengthseq( cand_targs) > 1) {
		/* need to select a source */
		DisplayWindow( FieldChoiceWindow);
		target = NIL;
		target = select_one_field( cand_targs, target, FieldChoiceWindow);
		}
	else target = carseq( cand_targs);


	/*******************************************************************/
	/*            Generate, Display, and Choose Source Field           */
	/*******************************************************************/
	/* generate candidate source fields */
	cand_srcs = NIL;
	cand_srcs = gen_cand_srcs( S, target, target_type, cand_srcs);
	if (nullseq( cand_srcs)) {
		printf("ERROR: No source fields.\n");
		return;
		}

	/* fill the FieldBuffer with the candidate source fields */
	/* initialize the FieldBuffer */
	InitializeTextBuffer( FB);
	StringMove( FB, "------- Please Scroll Down -------");
	NextLine( FB);
	StringMove( FB, "The source fields are: ");
	NextLine( FB);
	FillFieldBuffer( S, FB, cand_srcs);
	StringMove( FB, "***** End of Text *****");
	NextLine( FB);
	DisplayBuffer( FB, FieldScrollWindow);

	/* select one source field for access */
	if (lengthseq( cand_srcs) > 1) {
		/* need to select a source */
		DisplayWindow( FieldChoiceWindow);
		source = NIL;
		source = select_one_field( cand_srcs, source, FieldChoiceWindow);
		}
	else source = carseq( cand_srcs);

	/*******************************************************************/
	/*                    Generate an Analogical Map                   */
	/*******************************************************************/
	map = AllocateMap();
	map = make_map_from_net( S, map, target, source);

	/*******************************************************************/
	/*               Generate Metaphorical Surface Structures          */
	/*******************************************************************/
	if (GetMapLength(map) == NIL) {
		printf("No analogies between target and source.\n");
		printf("Analogical elaboration is not possible.\n");
		return;
		}
	metaphorical_surface_structs( S, map, target, source);

	free( FB);
	free( map);
}


