/***********************************************************************/
/*                                                                     */
/*                        Hypothesis Network Code                      */
/*                   Copyright (C) 1993 Eric Steinhart                 */
/*                                                                     */
/***********************************************************************/

#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\stdlib.h"
#include "anadefs.h"
#include "semnet.h"
#include "buffer.h"
#include "iterate.h"

extern int IncludeFrames;

#define MAX_HYPOS       50      /* maximum nodes in hypo net */
#define EXCITORY        1       /* weight of excitory connection */
#define INHIBITORY      -1      /* weight of inhibitory connection */

/* The Hypothesis Network */
struct HYPOTHESIS_NET {
	int num_hypos;                  /* number of match hypotheses */
	float seminput[MAX_HYPOS];      /* semantic input to units */
	float netinput[MAX_HYPOS];      /* net input to each unit */
	int w[MAX_HYPOS][MAX_HYPOS];    /* the weight matrix */
	float activation[MAX_HYPOS];   /* activations of units */
	sequence_ptr match_hypos;
	};

/* a stack of containment chains */
/* a containment chain is a sequence, so is a stack of sequences */
#define MAX_CHAINS 10
struct stack_of_chains {
	int num_chains;
	sequence_ptr ListOfChains[MAX_CHAINS];
	};

/* define global variables for tallying number of connections */
int TotalExcitory;
int TotalInhibitory;


/**************************************************************************/
/*                                                                        */
/*                             PRINTING FUNCTIONS                         */
/*                                                                        */
/**************************************************************************/

void PrintHypoNames( S, H)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
{
	cons_ptr x;
	sequence_ptr index;

	_clearscreen( _GCLEARSCREEN);
	printf("--------- Hypothesis Activations ---------\n");

	/* for each match hypothesis */
	SequenceIterator( index, H->match_hypos) {
		x = carseq( index);
		printf("(%-15.15s = %-15.15s)\n",
			GetName(S,first(x)),
			GetName(S,second(x)));
		}
}


void FillHypoConnBuffer( S, H, CB)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
struct TextBuffer *CB;
{
	cons_ptr x,y;
	sequence_ptr index;
	int i, j;

	/* print H->match_hypos */
	SequenceIterator( index, H->match_hypos) {
		x = carseq( index);
		StringMove(CB, "Match hypothesis (");
		StringMove(CB, GetName(S,first(x)));
		StringMove(CB, " = ");
		StringMove(CB, GetName(S,second(x)));
		StringMove(CB, ") has");
		NextLine(CB);

		/* print the connections of x */
		/* find the position of x in the match hypotheses */
		i = get_positionseq( x, H->match_hypos) - 1;
		/* for each hypothesis */
		for (j=0; j < H->num_hypos; j++) {
			y = get_nth_seq( H->match_hypos, j);
			if (H->w[i][j] == EXCITORY) {
			   StringMove(CB, "     EXCITATORY with (");
			   StringMove(CB, GetName(S,first(y)));
			   StringMove(CB, " = ");
			   StringMove(CB, GetName(S,second(y)));
			   StringMove(CB, ")");
			   NextLine(CB);
			   }
			if (H->w[i][j] == INHIBITORY) {
			   StringMove(CB, "     INHIBITORY with (");
			   StringMove(CB, GetName(S,first(y)));
			   StringMove(CB, " = ");
			   StringMove(CB, GetName(S,second(y)));
			   StringMove(CB, ")");
			   NextLine(CB);
			   }
			}
		}
}

void PrintSemInputs( S, H)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
{
	sequence_ptr HypoPtr;
	cons_ptr TheHypo;
	int i;

	/* print semantic inputs */
	_clearscreen( _GCLEARSCREEN);
	printf("----------- Initial Activations for Hypotheses -------\n");
	HypoPtr = H->match_hypos;
	for (i = 0; i < H->num_hypos; i++) {
		TheHypo = carseq( HypoPtr);
		printf("(%-15.15s = %-15.15s)      %f\n",
			GetName(S,first(TheHypo)),
			GetName(S,second(TheHypo)),
			H->seminput[i]);
		HypoPtr = cdrseq( HypoPtr);
		}
	strike_any_key();
}

void print_match_hypotheses( S, MH)
struct SEMANTIC_NETWORK *S;
sequence_ptr MH;
{
	sequence_ptr index;
	cons_ptr x;

	/* print match hypotheses */
	printf("The match hypotheses are: \n");
	printf("(");
	SequenceIterator( index, MH) {
		x = carseq( index);
		printf(" (%s,%s)\n", GetName(S,first(x)), GetName(S,second(x)));
		}
	printf(")\n");
	strike_any_key();
}

/**************************************************************************/
/*                                                                        */
/*                     HYPOTHESIS NET ALLOCATE & INIT                     */
/*                                                                        */
/**************************************************************************/
struct HYPOTHESIS_NET *AllocateHypothesisNet( void)
{
	struct HYPOTHESIS_NET *HypoNetPtr;

	HypoNetPtr = (struct HYPOTHESIS_NET *)
					malloc( (size_t) sizeof( struct HYPOTHESIS_NET));
	if (HypoNetPtr == NULL) {
		printf("ERROR: Can't allocate hypothesis network.\n");
		printf("ERROR: Out of memory.\n");
		exit( ERROR_VALUE);
		}
	return( HypoNetPtr);
}


struct HYPOTHESIS_NET *InitializeHypothesisNet( H)
struct HYPOTHESIS_NET *H;
{
	int i, j;

	/* set number of excitory and inhibitory connections to zero */
	TotalExcitory = 0;
	TotalInhibitory = 0;

	/* set number of hypotheses to zero */
	H->num_hypos = 0;

	/* set the seminput vector to floating zero */
	for (i=0; i < MAX_HYPOS; i++) H->seminput[i] = 0.0;

	/* set the weight matrix to zero */
	for (i=0; i < MAX_HYPOS; i++)
		for (j=0; j < MAX_HYPOS; j++)
			H->w[i][j] = 0;

	/* set the activation vector to floating zero */
	for (i=0; i < MAX_HYPOS; i++) H->activation[i] = 0.0;

	/* set the sequence_ptr of match hypotheses to null */
	H->match_hypos = NIL;

	return( H); /* return the initialized network */
}



/**************************************************************************/
/*                                                                        */
/*                    Initial Activations for Hypotheses                  */
/*                                                                        */
/**************************************************************************/
/* SetRawInputUnits adds one raw input to the seminput of each */
/* match hypotheses in AnalogousPairs */
struct HYPOTHESIS_NET *SetRawInputUnits( H, AnalogousPairs)
struct HYPOTHESIS_NET *H;
sequence_ptr AnalogousPairs;        /* ((T1, S1), (T2, S2), . . . (Tn, Sn)) */
{
	sequence_ptr apx;   /* indexes AnalogousPairs */
	cons_ptr x;         /* an analogous pair */
	int xpos;           /* position of a match hypothesis */

/*  printf("ENTER: SetRawInputUnits.\n");*/

	SequenceIterator( apx, AnalogousPairs) {
		x = carseq( apx); /* get an analogous pair */
		if (not( is_memberseq( x, H->match_hypos))) {
			printf("ERROR: Match hypothesis not found.\n");
			exit( ERROR_VALUE);
			}
		/* find the position of x in the match hypotheses */
		xpos = get_positionseq( x, H->match_hypos) - 1;
		/* increment number of raw input units to x */
		H->seminput[xpos] += 1.0;
		}
/*  printf("EXIT: SetRawInputUnits.\n");*/
	return( H); /* return the modified network */
}

struct HYPOTHESIS_NET *NormalizeInputs( H)
struct HYPOTHESIS_NET *H;
{
	float MaxRawInputs;
	int i;

/*  printf("ENTER: NormalizeInputs.\n");*/
	/* find maximum number of raw input units */
	MaxRawInputs = 0;
	for (i=0; i < MAX_HYPOS; i++)
		if (H->seminput[i] > MaxRawInputs)
			MaxRawInputs = H->seminput[i];

	/* divide every input by MaxRawInputs */
	for (i=0; i < MAX_HYPOS; i++)
		if (MaxRawInputs != 0.0)
			H->seminput[i] /= MaxRawInputs;
/*  printf("EXIT: NormalizeInputs.\n");*/
	return( H);
}



/**************************************************************************/
/*                                                                        */
/*                      PROCESS HYPOTHESIS NET CONNECTIONS                */
/*                                                                        */
/**************************************************************************/
/* AddExcitory takes a list of mutually supporting match hypotheses    */
/* and adds them and their excitory interconnections to the hypothesis */
/* network as follows:                                                 */
/*      for each x in MSMH                                             */
/*              add x to H->match_hypos                                */
/*      for each x in MSMH                                             */
/*              make excitory connection btwn x and                    */
/*              every other y in MSMH                                  */
struct HYPOTHESIS_NET *AddExcitory( S, H, MSMH)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
sequence_ptr MSMH;  /* mutually supportive match hypotheses.  Each item in */
		/* MSMH is a match hypothesis, i.e., a pair. */
{
	sequence_ptr index; /* indexes the sequences of match hypotheses */
	cons_ptr x;     /* a match hypothesis */
	int xpos;       /* position of x in H->match_hypos */
	sequence_ptr jdex;  /* indexes the sequence_ptr of match hypotheses */
	cons_ptr y;     /* a match hypothesis */
	int ypos;       /* position of y in H->match_hypos */

/*  printf("ENTER: AddExcitory.\n");*/
	SequenceIterator( index, MSMH) {
		x = carseq( index); /* x is a match hypothesis */
		/* append x to H->match_hypos */
		if (not( is_memberseq( x, H->match_hypos))) {
/*          printf("Entering (%s,%s) into hypothesis net.\n",
			GetName(S,first(x)),GetName(S,second(x)));*/
			H->match_hypos = appendseq( H->match_hypos, nconseq(x));
			H->num_hypos++; /* added a hypothesis */
			}
		}
/*  print_match_hypotheses(S, H->match_hypos);*/

	/* for each x in MSMH */
	/*      make an excitory connecton between x and */
	/*      every other y in MSMH */
/*  printf("Add the excitory connections.\n");*/
	SequenceIterator( index, MSMH) {
		x = carseq( index); /* x is a match hypothesis */
		xpos = get_positionseq( x, H->match_hypos);
		if (xpos == 0) {
			printf("ERROR:Element not in match hypotheses.\n");
			exit( ERROR_VALUE);
			}
		/* decrement x since get_positionseq begins with 1 */
		xpos--;
/*      printf("Position of (%s,%s) in match hypotheses is %d\n",
			GetName(S,first(x)),GetName(S,second(x)),
			xpos); */
		/* for every other y in MSMH */
		SequenceIterator( jdex, MSMH) {
			y = carseq( jdex); /* y is a match hypothesis */
/*          printf("Compare (%s,%s) with (%s,%s)\n",
			  GetName(S,first(x)),GetName(S,second(x)),
			  GetName(S,first(y)),GetName(S,second(y)));*/
			if (not (equal( y, x))) {
/*              printf("Match hypotheses are not equal\n");*/
				ypos = get_positionseq( y, H->match_hypos);
				if (ypos == 0) {
				  printf( "ERROR:Element not in match hypotheses.\n");
				  exit( ERROR_VALUE);
				  }
/*          printf(
			"Position of (%s,%s) in match hypotheses is %d\n",
					GetName(S,first(y)),
					GetName(S,second(y)),
					ypos);
				printf("x= (%s,%s) xpos = %d\n",
					GetName(S,first(x)),
					GetName(S,second(x)),
					xpos);
				printf("y= (%s,%s) ypos = %d\n",
					GetName(S,first(y)),
					GetName(S,second(y)),
					ypos);*/
				/* decrement ypos since */
				/* get_positionseq begins with 1 */
				ypos--;
				if (xpos == ypos) {
				   printf("ERROR:Self-excitation.\n");
				   exit( ERROR_VALUE);
				   }
				/* add excitory connection btwn x and y */
				H->w[xpos][ypos] = EXCITORY;
				/* add excitory connnection btwn y and x */
				H->w[ypos][xpos] = EXCITORY;
				} /* end if not equal x and y */
/*          else printf("Match hypotheses are equal.\n");*/
			} /* end sequence iterator with jdex */
		} /* end sequence iterator with index */

/*      printf("EXIT: AddExcitory.\n");*/
	return( H);
}

/* AddInhibitory finds the match hypotheses in the hypothesis net */
/* that are mutually inconsistent and adds inhibitory links to the net */
/* The list of match hypotheses must contain distinct items, that is, it */
/* cannot contain two pairs x and y such that x = y */
struct HYPOTHESIS_NET *AddInhibitory( S, H)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
{
	sequence_ptr XMH;   /* indexes match hypotheses */
	cons_ptr x;         /* a match hypothesis */
	sequence_ptr YMH;   /* indexes match hypotheses */
	cons_ptr y;         /* a match hypothesis */
	int xpos, ypos;

/*  printf("ENTER: AddInhibitory.\n");*/
	SequenceIterator( XMH, H->match_hypos) {
		x = carseq( XMH);       /* a match hypothesis */
		SequenceIterator( YMH, H->match_hypos) {
		    y = carseq( YMH);
/*          printf("Compare (%s,%s) with (%s,%s)\n",
				GetName(S,first(x)),
				GetName(S,second(x)),
				GetName(S,first(y)),
				GetName(S,second(y))); */
		    if (not( equal( x, y)))
			  if ((first(x) == first(y)) ||
				(second(x) == second(y)) ) {
			    /* then x and y share one but not both items */
			    /* and are mutually inconsistent */
/*              printf("Inconsistent: (%s,%s) and (%s,%s)\n",
					GetName(S,first(x)),
					GetName(S,second(x)),
					GetName(S,first(y)),
					GetName(S,second(y)));*/
			    xpos = get_positionseq( x, H->match_hypos);
			    ypos = get_positionseq( y, H->match_hypos);
			    if ((xpos == 0) || (ypos == 0)) {
				   printf("ERROR:Element not in match hypotheses.\n");
			       exit( ERROR_VALUE);
			       }
			    /* decrement positions, since */
			    /* get_positionseq begins with 1 */
			    xpos--; ypos--;
			    if (xpos == ypos) {
			       printf("ERROR: Self-inhibition.\n");
			       exit( ERROR_VALUE);
			       }
			    /* add an inhibitory link between x and y */
			    H->w[xpos][ypos] = INHIBITORY;
			    /* add the inverse link */
			    H->w[ypos][xpos] = INHIBITORY;
			    } /* end if inconsistent */
			} /* end sequence iterator with YMH */
		} /* end sequence iterator with XMH */

/*  printf("EXIT: AddInhibitory.\n");*/
	return( H);
}

/* counts the number of excitory and inhibitory connections */
/* sets global variables TotalExcitory and TotalInhibitory */
void CountConnections( H)
struct HYPOTHESIS_NET *H;
{
	int i, j;

	for (i=0; i < H->num_hypos; i++) {
	    for (j=0; j < H->num_hypos; j++) {
		if (H->w[i][j] == EXCITORY) TotalExcitory++;
		if (H->w[i][j] == INHIBITORY) TotalInhibitory++; } }
}


/**************************************************************************/
/*                      Principle of Proportional Analogy                 */
/**************************************************************************/
/* make pairs of analogous frames.  Analogous frames share the */
/* same verb.  Verb must be in TargetVerbs */
/* if (x,y) is a pair of analogous frames, x is in Target & y in Source */
sequence_ptr MakePairsOfFrames( S, V, Target, Source, PairsOfFrames)
struct SEMANTIC_NETWORK *S;
int V;                          /* the verb */
cons_ptr Target;
cons_ptr Source;
sequence_ptr PairsOfFrames;
{
	cons_ptr pframes;       /* frames in which V participates */
	cons_ptr TargetFrames;
	cons_ptr SourceFrames;
	cons_ptr pfx;           /* indexes participating frames */
	int this_frame;
	cons_ptr tfx;           /* indexes TargetFrames */
	cons_ptr sfx;           /* indexes SourceFrames */
	int this_target_frame;
	int this_source_frame;

/*  printf("ENTER: MakePairsOfFrames.\n");*/

/*  printf("Find participating frames of %s\n", GetName(S,V));*/

	/* propagate activation from V to participating frames */
	/* NOTICE! These frames are actually frame identifier nodes */
	pframes = NIL;
	pframes = node_trans( V, S, VERB, LITERAL, pframes);
	if (null(pframes)) {
		printf("ERROR: Verb <%s> induces no analogous frames.\n",
			GetName(S,V));
		exit(ERROR_VALUE);
		}
/*  printf("Participating frames are:\n");
	print_node_names( pframes, S); printf("\n");*/

	/* partition participating frames into TargetFrames & SourceFrames */
	/* NOTICE that this process ignores frames not in Target or Source */
/*  printf("Partitioning frames into TargetFrames & SourceFrames.\n");*/
	TargetFrames = NIL;
	SourceFrames = NIL;
	ListIterator( pfx, pframes) {
		this_frame = car( pfx);
		if (is_member( this_frame, Target))
			TargetFrames = cons( this_frame, TargetFrames);
		if (is_member( this_frame, Source))
			SourceFrames = cons( this_frame, SourceFrames);
		}
/*  printf("TargetFrames is: \n");
	print_node_names( TargetFrames, S); printf("\n");
	printf("SourceFrames is: \n");
	print_node_names( SourceFrames, S); printf("\n");*/

	/* make pairs of frames from TargetFrames & SourceFrames */
	/* each pair is (TargetFrame, SourceFrame) */
/*  printf("Make (TargetFrame, SourceFrame) pairs.\n");*/
	ListIterator( tfx, TargetFrames) {
		this_target_frame = car( tfx);
		ListIterator( sfx, SourceFrames) {
			this_source_frame = car( sfx);
			PairsOfFrames = unionseq( PairsOfFrames,
					   nconseq( cons( this_target_frame,
					      ncons( this_source_frame))));
			}
		}
	FreeList( TargetFrames);
	FreeList( SourceFrames);
/*  printf("EXIT: MakePairsOfFrames.\n");*/
	return( PairsOfFrames);
}

/* Makes the sequences of AnalogousPairs. AnalogousPairs contains */
/* all the semantic analogies induced by a pair of frames P */
/* the frames in P are analogous because they share the same verb */
/* AnalogousPairs has form ((TA1,SA1),(TA2,SA2), . . ,(TAn,SAn)) */
/* where each (TAi,SAi) is an analogous pair induced by P */
/* Given a pair of analogous frames (F1, F2) where */
/*      TargetFrame F1: AGENT:student, PATIENT:idea */
/*      SourceFrame F2: AGENT:mother, PATIENT:baby */
/* AnalogousPairs is ((student, mother), (idea, baby)) */
sequence_ptr MakeAnalogousPairs( S, AB, P, AnalogousPairs)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *AB;
cons_ptr P;     /* a pair of analogous frames */
sequence_ptr AnalogousPairs;
{
	int this_role;
	cons_ptr TargetAnalogues;
	int TA;
	cons_ptr SourceAnalogues;
	int SA;

/*  printf("ENTER: MakeAnalogousPairs.\n");*/
	/* for every non-verb functional thematic role */
	for (this_role = 1; this_role < MAX_FUNC_ROLES; this_role++) {
		TargetAnalogues = NIL;
		TargetAnalogues = node_trans( first(P), S, this_role,
										LITERAL, TargetAnalogues);
		if (length( TargetAnalogues) > 1) {
		   printf("ERROR: Multiple target analogues.\n");
		   print_node_names( TargetAnalogues, S); printf("\n");
		   exit( ERROR_VALUE);
		   }

		SourceAnalogues = NIL;
		SourceAnalogues = node_trans( second(P), S, this_role,
					LITERAL, SourceAnalogues);
		if (length( SourceAnalogues) > 1) {
		   printf("ERROR: Multiple source analogues.\n");
		   exit( ERROR_VALUE);
		   }

		if (not(null(TargetAnalogues)) &&
			not(null(SourceAnalogues))) {
		   TA = car( TargetAnalogues);
		   SA = car( SourceAnalogues);
		   StringMove( AB, "       Role ");
		   StringMove( AB, role_names[this_role]);
		   StringMove( AB, " induces (");
		   StringMove( AB, GetName(S,TA));
		   StringMove( AB, " = ");
		   StringMove( AB, GetName(S,SA));
		   StringMove( AB, ")");
		   NextLine(AB);
		   AnalogousPairs = unionseq( AnalogousPairs,
				    nconseq( cons( TA, ncons( SA))));
		   }
		}
/*  printf("EXIT: MakeAnalogousPairs.\n");*/
	return( AnalogousPairs);
}



/**************************************************************************/
/*          Principle of Extension to Paradigmatic Contrast Sets          */
/**************************************************************************/

int ContrastiveRelation( S, PFrame)
struct SEMANTIC_NETWORK *S;
int PFrame;
{
	cons_ptr TheRelation;

/*  printf("ContrastiveRelation(<%s>) is: ", GetName(S,PFrame));*/
	TheRelation = NIL;
	TheRelation = node_trans( PFrame, S,
							   CONTRAST, LITERAL, TheRelation);
/*  print_node_names( TheRelation, S); printf("\n");*/

	if (length(TheRelation) > 1) {
		printf("ERROR:<%s> has more than one contastive relation.\n",
				GetName(S,PFrame));
		exit( ERROR_VALUE);
		}
	else return( car( TheRelation));
}

struct HYPOTHESIS_NET *ExtendToParaContrastSets( S, AB, H, AnalogousPairs)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *AB;
struct HYPOTHESIS_NET *H;
sequence_ptr AnalogousPairs;
{

	sequence_ptr AnaPairPtr;        /* indexes AnalogousPairs */
	cons_ptr Pair;                  /* (T,S) */
	int TopicTerm;                  /* Topic adjective */
	int SrcTerm;                    /* Source adjective */
	cons_ptr TopicParaFrame;        /* PFRAMES of T */
	cons_ptr SourceParaFrame;       /* PFRAMES of S */
	cons_ptr TopicAnalogue;
	cons_ptr SourceAnalogue;
	int P;                          /* a paradigmatic role */
	cons_ptr AdjectivalPair;        /* (TopicAnalogue,SourceAnalogue) */

/*  printf("ENTER: ExtendToParaContrastSets\n");*/
	SequenceIterator( AnaPairPtr, AnalogousPairs) {
		Pair = carseq( AnaPairPtr);
		TopicTerm = first(Pair);
/*      printf("TopicTerm = <%s>\n", GetName(S,TopicTerm));*/
		SrcTerm = second(Pair);
/*      printf("SrcTerm = <%s>\n", GetName(S,SrcTerm));*/
		if (is_adjective(S, TopicTerm) & is_adjective(S, SrcTerm)) {
			TopicParaFrame = NIL;
			TopicParaFrame = para_node_trans(TopicTerm,
									S, LITERAL, TopicParaFrame);
/*          printf("TopicParaFrame = ");
			print_node_names( TopicParaFrame, S); printf("\n");*/
			SourceParaFrame = NIL;
			SourceParaFrame = para_node_trans(SrcTerm,
									S, LITERAL, SourceParaFrame);
/*          printf("SourceParaFrame = ");
			print_node_names( SourceParaFrame, S); printf("\n");*/
			if (ContrastiveRelation(S, first(TopicParaFrame)) ==
				 ContrastiveRelation(S, first(SourceParaFrame))) {
				for (P=FIRST_PARA_ARG_ROLE; P<=LAST_PARA_ROLE; P++) {
					TopicAnalogue = NIL;
					TopicAnalogue = node_trans(first(TopicParaFrame),
											S, P, LITERAL, TopicAnalogue);
					SourceAnalogue = NIL;
					SourceAnalogue = node_trans(first(SourceParaFrame),
											S, P, LITERAL, SourceAnalogue);
					AdjectivalPair = cons(first(TopicAnalogue),
										 ncons(first(SourceAnalogue)));
					if (not(null(AdjectivalPair))) {
						StringMove( AB, "       EXTENDED to (");
						StringMove( AB, GetName(S,car(TopicAnalogue)));
						StringMove( AB, " = ");
						StringMove( AB, GetName(S,car(SourceAnalogue)));
						StringMove( AB, ")");
						NextLine( AB);
						}
					H = AddExcitory(S, H, conseq( Pair,
											nconseq(AdjectivalPair)));
					} /* for each paradigmatic role */
				} /* if equal contrastive relations */
			} /* if (adjective, adjective) */
		 } /* end SequenceIterator with AnaPairPtr */
/*  printf("EXIT: ExtendToParaContrastSets\n");*/
	return( H);
}


/**************************************************************************/
/*                 Principle of Extension to Covering Sets                */
/**************************************************************************/
int FindOrigin( S, SisterTerms)
struct SEMANTIC_NETWORK *S;
cons_ptr SisterTerms;
{
	cons_ptr ptr;
	int term, ThePredecessor;

	/* find the term in SisterTerms that has no predecessor */
	/* for each term in SisterTerms do {                    */
	/*      if not(term has a predecessor) return( term);}  */

	if (null(SisterTerms)) return( NO_FILLER);

	ListIterator( ptr, SisterTerms) {
		term = car(ptr);
		ThePredecessor = GetPredecessorOf(S, term);
		if (ThePredecessor == NO_FILLER) return(term);
		}
	return(NO_FILLER);
}

struct HYPOTHESIS_NET *ExtendToCoveringSets(
							S, AB, H, TargetTerm, SourceTerm)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *AB;
struct HYPOTHESIS_NET *H;
int TargetTerm;
int SourceTerm;
{
	cons_ptr TargetSubtypes, SourceSubtypes;
	int TargetOrigin, SourceOrigin;
	int CurrentTargetTerm, CurrentSourceTerm;
	cons_ptr ThisPair;
	sequence_ptr AnalogousPairs;

	TargetSubtypes = NIL;
	TargetSubtypes = node_trans(TargetTerm, S,
						SUBTYPE, LITERAL, TargetSubtypes);

	SourceSubtypes = NIL;
	SourceSubtypes = node_trans(SourceTerm, S,
						SUBTYPE, LITERAL, SourceSubtypes);

	TargetOrigin = FindOrigin(S, TargetSubtypes);
	SourceOrigin = FindOrigin(S, SourceSubtypes);

	CurrentTargetTerm = TargetOrigin;
	CurrentSourceTerm = SourceOrigin;
	AnalogousPairs = nconseq(cons(TargetTerm, ncons(SourceTerm)));
	while ((CurrentTargetTerm != NO_FILLER) &&
				(CurrentSourceTerm != NO_FILLER)) {
		ThisPair = cons( CurrentTargetTerm, ncons(CurrentSourceTerm));
		AnalogousPairs = conseq(ThisPair, AnalogousPairs);
		CurrentTargetTerm = GetSuccessorOf(S, CurrentTargetTerm);
		CurrentSourceTerm = GetSuccessorOf(S, CurrentSourceTerm);
		}

	H = AddExcitory(S, H, AnalogousPairs);
	return(H);
}


/**************************************************************************/
/*                  Principle of Extension to Properties                  */
/**************************************************************************/
struct HYPOTHESIS_NET *ExtendToProperties( S, AB, H, AnalogousPairs)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *AB;
struct HYPOTHESIS_NET *H;
sequence_ptr AnalogousPairs;
{

	/****************************************************************/
	/* for each (t,s) in AnalogousPairs do {                        */
	/*  if IsProperty(t) & IsProperty(s) then do {                  */
	/*      (GetAttributeTerm(t), GetAttributeTerm(s)) is a match   */
	/*      (GetObjectTerm(t), GetObjectTerm(s)) is a match         */
	/*      treat LiteralValuesOf(t) & LiteralValuesOf(s)           */
	/*          as analogous paradigmatic contrast sets             */
	/****************************************************************/

	sequence_ptr AnaPairPtr;        /* indexes AnalogousPairs */
	cons_ptr Pair;                  /* (T,S) */
	int TopicTerm;
	int SourceTerm;
	int TopicAttribute;
	int SourceAttribute;
	int TopicObject;
	int SourceObject;
	int P;                          /* a paradigmatic role */
	cons_ptr AttributePair;         /* (TopicAttribute,SourceAttribute) */
	cons_ptr ObjectPair;            /* (TopicObject,SourceObject) */

/*  printf("ENTER: ExtendToProperties\n");*/
	SequenceIterator( AnaPairPtr, AnalogousPairs) {
		Pair = carseq( AnaPairPtr);
		TopicTerm = first(Pair);
/*      printf("TopicTerm = <%s>\n", GetName(S,TopicTerm));*/
		SourceTerm = second(Pair);
/*      printf("SourceTerm = <%s>\n", GetName(S,SourceTerm));*/
		if ((GetType(S,TopicTerm) == PROPERTYTYPE) &&
				(GetType(S,SourceTerm) == PROPERTYTYPE)) {

			/**************** PROCESS ATTRIBUTES OF THE PROPERTIES ******/
			TopicAttribute = GetAttributeTerm(S, TopicTerm);
			if (TopicAttribute == NO_FILLER) {
				_clearscreen( _GCLEARSCREEN);
				printf("ERROR: No attribute for property frame <%s>.\n",
						GetName(S,TopicTerm));
				exit(ERROR_VALUE);
				}

			SourceAttribute = GetAttributeTerm(S, SourceTerm);
			if (SourceAttribute == NO_FILLER) {
				_clearscreen( _GCLEARSCREEN);
				printf("ERROR: No attribute for property frame <%s>.\n",
						GetName(S,SourceTerm));
				exit(ERROR_VALUE);
				}

			AttributePair = cons(TopicAttribute, ncons(SourceAttribute));
			if (not(null(AttributePair))) {
				StringMove( AB, "       EXTENDED to (");
				StringMove( AB, GetName(S,TopicAttribute));
				StringMove( AB, ", ");
				StringMove( AB, GetName(S,SourceAttribute));
				StringMove( AB, ")");
				NextLine( AB);
				}

			/************* PROCESS OBJECTS OF THE PROPERTIES ***********/
			TopicObject = GetObjectTerm(S, TopicTerm);
			if (TopicObject == NO_FILLER) {
				_clearscreen( _GCLEARSCREEN);
				printf("ERROR: No object for property frame <%s>.\n",
						GetName(S,TopicTerm));
				exit(ERROR_VALUE);
				}

			SourceObject = GetObjectTerm(S, SourceTerm);
			if (SourceObject == NO_FILLER) {
				_clearscreen( _GCLEARSCREEN);
				printf("ERROR: No object for property frame <%s>.\n",
						GetName(S,SourceTerm));
				exit(ERROR_VALUE);
				}

			ObjectPair = cons(TopicObject, ncons(SourceObject));
			if (not(null(ObjectPair))) {
				StringMove( AB, "       EXTENDED to (");
				StringMove( AB, GetName(S,TopicObject));
				StringMove( AB, ", ");
				StringMove( AB, GetName(S,SourceObject));
				NextLine( AB);
				}
			H = AddExcitory(S, H, conseq( Pair,
									conseq( ObjectPair,
									  nconseq(AttributePair))));
			H = ExtendToCoveringSets(S, AB, H,
					TopicAttribute, SourceAttribute);
			} /* if (property, property) */
		 } /* end SequenceIterator with AnaPairPtr */
/*  printf("EXIT: ExtendToProperties\n");*/
	return( H);
}




/**************************************************************************/
/*                       Principle of Structural Analogy                  */
/**************************************************************************/
int ValidWhole( W, Field)
int W;          /* the whole to be tested */
cons_ptr Field; /* the field in which W must lie */
{
	/* to be valid, a whole must not be NO_WHOLE */
	/* and must be a member of the given field */
	if ((W != NO_WHOLE) && is_member(W, Field))
		return( TRUE); /* the whole is valid */
	else return( FALSE);   /* the whole is NOT valid */
}


/* StructuralAnalogies returns a sequence of mutually consistent */
/* match hypotheses.  Such a sequence is a containment chain. */
/* This procedure climbs up the mereological hierarchy in accordance  */
/* with the structural analogy principle */
sequence_ptr StructuralAnalogies( S, P, Target, Source, result)
struct SEMANTIC_NETWORK *S;
cons_ptr P;             /* an analogous pair */
cons_ptr Target;        /* the Target field */
cons_ptr Source;        /* the Source field */
sequence_ptr result;    /* the containment chain generated by P */
{
	int a;  /* first item in an analogous pair; a is in Target */
	int b;  /* second item in an analogous pair; b is in Source */

/*  printf("ENTER: StructuralAnalogies.\n");*/
	result = nconseq(P);
	a = first(P);
	b = second(P);
	while( ValidWhole( WholeOf(S,a), Target) &&
	       ValidWhole( WholeOf(S,b), Source)) {
		result = appendseq( result, nconseq(
			    cons( WholeOf(S,a), ncons( WholeOf(S,b)))));
		a = WholeOf(S,a); /* up mereological hierarchy in Target */
		b = WholeOf(S,b); /* up mereological hierarchy in Source */
		}

/*  printf("EXIT: StructuralAnalogies.\n");*/
	return( result);
}


/* chain is a containment chain.  Containment chains are ordered. */
/* The pair (u,v) precedes (WholeOf(u),WholeOf(v)).  The members of a */
/* containment chain are match hypotheses and are mutually supportive. */
/* GenerateChains merges chains in accordance with the chain */
/* reinforcement principle */
struct stack_of_chains *GenerateChains(
							S, AB, AnalogousPairs, Target, Source, SOC)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *AB;
sequence_ptr AnalogousPairs;
cons_ptr Target;
cons_ptr Source;
struct stack_of_chains *SOC;
{
	sequence_ptr apx;       /* indexes AnalogousPairs */
	cons_ptr P;             /* an analogous pair */
	sequence_ptr Chain;     /* a containment chain */
	sequence_ptr newchain;  /* a containment chain */
	int i;
	int HasBeenMerged;      /* chain merge flag */
	cons_ptr x, y;
	sequence_ptr cptr;
	cons_ptr ProportionalMatch, StructuralMatch;

/*  printf("ENTER: GenerateChains.\n");*/
	SequenceIterator( apx, AnalogousPairs) {
		P = carseq( apx); /* get an analogous pair */
		Chain = NIL;
		Chain = StructuralAnalogies( S, P, Target, Source, Chain);

		if (lengthseq( Chain) > 1) {
			ProportionalMatch = carseq(Chain);
			StringMove( AB, "     Proportional Match: (");
			StringMove( AB, GetName(S,first(ProportionalMatch)));
			StringMove( AB, " = ");
			StringMove( AB, GetName(S,second(ProportionalMatch)));
			StringMove( AB, ")");
			NextLine( AB);
			SequenceIterator( cptr, Chain) {
				StructuralMatch = carseq(cptr);
				StringMove( AB, "     Structural Match: (");
				StringMove( AB, GetName(S,first(StructuralMatch)));
				StringMove( AB, " = ");
				StringMove( AB, GetName(S,second(StructuralMatch)));
				StringMove( AB, ")");
				NextLine( AB);
				}
			}

		/* if the last element of chain = the last element */
		/* of any member of the stack of chains, */
		/* then chain gets merged with that chain on the stack */
		/* else chain gets pushed onto the stack */
/*      printf("See if chain can be merged.\n");*/
		HasBeenMerged = FALSE;
/*      printf("Number of chains on stack: %d\n",SOC->num_chains);*/
		for (i=0; i < SOC->num_chains; i++) {
/*          printf("Find carseq(lastseq(Chain))\n");*/
		    x = carseq(lastseq(Chain));
/*          printf("Find carseq(lastseq(SOC->ListOfChains[i]))\n");*/
		    y = carseq(lastseq(SOC->ListOfChains[i]));
		    if (equal(x,y)) {
/*             printf("last(Chain)=last(ListOfChains[%d]\n", i);*/
		       HasBeenMerged = TRUE;
		       newchain = unionseq( Chain, SOC->ListOfChains[i]);
		       SOC->ListOfChains[i] = newchain;
		       }
		    } /* end for i */

		if (not(HasBeenMerged)) {
		       /* push the chain onto the chain stack */
/*             printf("Push chain onto the chain stack.\n");*/
		       if (SOC->num_chains > MAX_CHAINS) {
				  printf("ERROR: Chain stack overflow.\n");
				  exit( ERROR_VALUE);
				  }
		       SOC->ListOfChains[SOC->num_chains] = Chain;
		       SOC->num_chains++;
		       }
		}

/*  printf("EXIT: GenerateChains.\n");*/
	return( SOC);
}


/* result = { y in Z | first(y) == x } */
sequence_ptr FirstCompatible( Z, x, result)
sequence_ptr Z;         /* a list of match hypotheses */
int x;                  /* a node in the network */
sequence_ptr result;    /* the result */
{

	sequence_ptr Zptr;
	cons_ptr y;             /* y is a candidate match hypo */

	SequenceIterator( Zptr, Z) {
		y = carseq(Zptr); /* a candidate match hypo */
		if (first(y) == x)
			result = appendseq( result, nconseq(y));
		}
	return( result);
}


/**************************************************************************/
/*                 Principle of Transitive Reinforcement                  */
/**************************************************************************/
/* find mutually compatible semantic match hypotheses included in a */
/* containment chain.  Transitivity of the containment relation, and */
/* the fact that these are semantic match hypos, indicates that they */
/* should be mutually supporting. TransitiveReinforcement adds the relevant */
/* mutually excitory connections to the Hypothesis Network */
struct HYPOTHESIS_NET *TransitiveReinforcement( S, H, AB, SemanticAnalogies)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
struct TextBuffer *AB;
sequence_ptr SemanticAnalogies;   /* a sequence of Semantic Analogies */
{
	/*********************************************************/
	/* for each (t,s) in SemanticAnalogies do {              */
	/*     TargChain = ContChain( t);                        */
	/*     SrcChain = ContChain( s);                         */
	/*     for each x in TargChain do {                      */
	/*         SMH = { A in SemanticAnalogies | first(A)=x } */
	/*         for each y in SMH do {                        */
	/*             if (second(y) is in SrcChain)             */
	/*             then (t,s) and y are mutually supportive; */
	/*             }}}                                       */
	/*********************************************************/

	sequence_ptr SAptr;
	cons_ptr ts, TargChain, SrcChain, Tptr;
	int x;
	sequence_ptr SMH, SMHptr;
	cons_ptr y;

	/* for each (t, s) in SemanticAnalogies do */
	SequenceIterator( SAptr, SemanticAnalogies) {
		ts = carseq( SAptr); /* the pair (t, s) */

		StringMove( AB, "Seek containing proportional matches for (");
		StringMove( AB, GetName(S,first(ts)));
		StringMove( AB, " = ");
		StringMove( AB, GetName(S,second(ts)));
		StringMove( AB, ")");
		NextLine( AB);

		TargChain = NIL;
		TargChain = MereologicalChain( S, first(ts), TargChain);
/*      printf("Target Chain: ");
		print_node_names( TargChain, S); printf("\n");*/
		SrcChain = NIL;
		SrcChain = MereologicalChain( S, second(ts), SrcChain);
/*      printf("Source Chain: ");
		print_node_names( SrcChain, S); printf("\n");*/
		ListIterator( Tptr, TargChain) {
			x = car(Tptr); /* x is in TargChain */
			SMH = NIL;
			SMH = FirstCompatible(SemanticAnalogies, x, SMH);
/*          printf("The compatible match hypotheses are:\n");
			print_match_hypotheses(S, SMH);*/
			/* for each y in SMH do { */
			SequenceIterator( SMHptr, SMH) {
				y = carseq( SMHptr); /* y is a match hypo */
				if (is_member(second(y), SrcChain))
					/* add EXCITORY btwn ts & y */
					H = AddExcitory(S, H, conseq( ts, nconseq(y)));
				}
			}
		}
	return( H); /* return the modified hypothesis net */
}


/**************************************************************************/
/*                 Principle of Adjectival Support                        */
/**************************************************************************/
struct HYPOTHESIS_NET *AdjectivalSupport( S, H, PairsOfFrames)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
sequence_ptr PairsOfFrames;
{
	sequence_ptr PairsPtr;

	SequenceIterator( PairsPtr, PairsOfFrames) {
		cons_ptr ThisPair;
		int TargetFrame;
		int SourceFrame;
		int Position;

		ThisPair = carseq( PairsPtr);
		TargetFrame = first( ThisPair);
		SourceFrame = second( ThisPair);
/*      printf( "Frames are: <%s, %s>\n",
				GetName(S, TargetFrame), GetName(S, SourceFrame));*/

		for (Position=1; Position <= MAX_SURFACE_POSITIONS; Position++) {
			int TargetNoun;
			cons_ptr TargetNounFeatures;
			int SourceNoun;
			cons_ptr SourceNounFeatures;
			cons_ptr AdjPtr;

			TargetNoun = GetNounAtPosition(S, TargetFrame, Position);
			if (TargetNoun != NO_FILLER) {
				TargetNounFeatures = NIL;
				TargetNounFeatures = GetFeatures( S, TargetNoun,
									LITERAL, TargetNounFeatures);

				SourceNoun = GetNounAtPosition(S, SourceFrame, Position);
				if (SourceNoun != NO_FILLER) {
					SourceNounFeatures = NIL;
					SourceNounFeatures = GetFeatures( S, SourceNoun,
									LITERAL, SourceNounFeatures);

					ListIterator( AdjPtr, TargetNounFeatures) {
						int ThisAdj;
						ThisAdj = car( AdjPtr);

						if (is_member( ThisAdj, SourceNounFeatures)) {
							printf("Nouns <%s, %s> share feature <%s>\n",
								GetName(S, TargetNoun),
								GetName(S, SourceNoun),
								GetName(S, ThisAdj));
							printf("Add one raw input unit to <%s, %s>\n",
								GetName(S, TargetFrame),
								GetName(S, SourceFrame));
							H = SetRawInputUnits( H, nconseq( ThisPair));
							}
						}
					}
				}
			}
		}
	return( H);
}


struct HYPOTHESIS_NET *BuildHypothesisNet( S, H, AB, TargetVerbs,
				Target, Source)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
struct TextBuffer *AB;
cons_ptr TargetVerbs;
cons_ptr Target;
cons_ptr Source;
{
	cons_ptr TV;                 /* points at TargetVerbs */
	int V;                       /* a target verb */
	sequence_ptr PairsOfFrames;  /* sequence of pairs of frames */
								 /* items in each pair are analogues */
								 /* first(P) is in Target */
								 /* second(P) is in Source */
	sequence_ptr pofx;           /* indexes PairsOfFrames */
	cons_ptr P;                  /* a pair of analogous frames */
	sequence_ptr AnalogousPairs; /* sequence_ptr of analogous words */
	struct stack_of_chains the_stack;   /* a stack of chains */
	struct stack_of_chains *ChainStack; /* pointer to stack of chains */
	sequence_ptr Chain;          /* a containment chain */
	int i;
	sequence_ptr SemanticAnalogies;

/*  printf("ENTER: BuildHypothesisNet.\n");*/

	/* SemanticAnalogies contains all match hypotheses induced */
	/* by verbs; it contains no purely structural analogies */
	SemanticAnalogies = NIL;

	/* allocate the ChainStack */
	ChainStack = &the_stack;

	/******************* GENERATE EXCITORY CONNECTIONS ****************/
/*  printf("Generating excitory connections.\n");*/

	ListIterator( TV, TargetVerbs) {
	  V = car(TV);    /* V is the target verb */

	  StringMove( AB, "The TargetVerb is: ");
	  StringMove( AB, GetName(S,V));
	  NextLine( AB);

	  /* make pairs of frames for the target verb */
	  PairsOfFrames = NIL;
	  PairsOfFrames = MakePairsOfFrames( S, V, Target, Source, PairsOfFrames);
/*    printf("The PairsOfFrames are:\n");
	  print_sequence_names( PairsOfFrames, S);*/

	  SequenceIterator( pofx, PairsOfFrames) {
		P = carseq( pofx); /* P is a pair of frames */

		StringMove( AB, "Analogous frames: (");
		StringMove( AB, GetName(S,first(P)));
		StringMove( AB, " = ");
		StringMove( AB, GetName(S,second(P)));
		StringMove( AB, ")");
		NextLine( AB);


		/**************** INDUCE SEMANTIC ANALOGIES ***************/
		DisplayExtraMsg( FieldChoiceWindow,
		  "Building hypothesis network: Inducing proportional analogies    ");

		StringMove( AB, "INDUCE PROPORTIONAL ANALOGIES.");
		NextLine( AB);

		/* STEP 1. Find AnalogousPairs.  AnalogousPairs contains */
		/* analogies induced by the frames in a pair of frames P */
/*      printf("STEP 1. Find AnalogousPairs.\n");*/
		if (IncludeFrames) AnalogousPairs = nconseq(P);
		else AnalogousPairs = NIL;
		AnalogousPairs = MakeAnalogousPairs( S, AB, P, AnalogousPairs);
		SemanticAnalogies = unionseq( AnalogousPairs, SemanticAnalogies);

		/* STEP 2. AnalogousPairs is a list of match hypotheses. */
		/* The match hypotheses in AnalogousPairs are all mutually */
		/* supporting. Add AnalogousPairs to the hypothesis net. */
/*      printf("STEP 2. Add excitory connections.\n");
		print_match_hypotheses( S, AnalogousPairs);*/
		H = AddExcitory( S, H, AnalogousPairs);

		/* STEP 3. Each x in AnalogousPairs gets a unit of raw */
		/* input since it is induced by a semantic relation */
/*      printf("STEP 3. Add raw input units.\n");*/
		H = SetRawInputUnits( H, AnalogousPairs);

		/*********** EXTEND TO PARADIGMATIC CONTRAST SETS **********/
		DisplayExtraMsg( FieldChoiceWindow,
		  "Building hypothesis network: Extending to contrast sets         ");

		StringMove( AB, "EXTEND TO PARADIGMATIC CONTRAST SETS.");
		NextLine( AB);
		H = ExtendToParaContrastSets( S, AB, H, AnalogousPairs);

		/******************* EXTEND TO PROPERTIES **********/
		StringMove( AB, "EXTEND TO PROPERTIES.");
		DisplayExtraMsg( FieldChoiceWindow,
		  "Building hypothesis network: Extending to properties            ");
		NextLine( AB);
		H = ExtendToProperties( S, AB, H, AnalogousPairs);

		/**************** INDUCE STRUCTURAL ANALOGIES **************/
		/* Use the semantic analogies to induce structural */
		/* analogies according to the structural analogy principle */
		DisplayExtraMsg( FieldChoiceWindow,
		  "Building hypothesis network: Inducing structural analogies      ");
		StringMove( AB, "INDUCE STRUCTURAL ANALOGIES.\n");
		NextLine( AB);

		/* STEP 1. Generate containment chains. */
/*      printf("STEP 1. Generate containment chains.\n");*/
		/* initialize the stack of chains */
		ChainStack->num_chains = 0;
		for (i=0; i < MAX_CHAINS; i++)
			ChainStack->ListOfChains[i] = NIL;
		ChainStack = GenerateChains(
						S, AB, AnalogousPairs, Target, Source, ChainStack);
		NextLine( AB);

		/* STEP 2. Add the new sets of mutually supporting match */
		/* hypotheses to the hypothesis network. */
/*      printf("STEP 2. Add containment chains to net.\n");*/
		for (i=0; i < ChainStack->num_chains; i++) {
			Chain = ChainStack->ListOfChains[i];
/*          printf("Add containment chain %d\n", i);*/
/*          print_match_hypotheses(S, Chain);*/
			H = AddExcitory( S, H, Chain);
			}

	  } /* P in pairs of frames */
	} /* end V in TargetVerbs */

	/******************* ADJECTIVAL SUPPORT **********************/
	H = AdjectivalSupport( S, H, PairsOfFrames);

	/******************* TRANSITIVE REINFORCEMENT **********************/
	DisplayExtraMsg( FieldChoiceWindow,
	  "Building hypothesis network: Looking for transitive reinforcements  ");
	StringMove( AB, "TRANSITIVE REINFORCEMENT.");
	NextLine( AB);
	H = TransitiveReinforcement( S, H, AB, SemanticAnalogies);
	NextLine( AB);

	/******************* GENERATE INHIBITORY CONNECTIONS ***************/
	DisplayExtraMsg( FieldChoiceWindow,
	  "Building hypothesis network: Generating inhibitory connections      ");

	StringMove( AB, "GENERATE INHIBITORY CONNECTIONS.");
	NextLine( AB);
	H = AddInhibitory( S, H);

	/******************* NORMALIZE SEMANTIC INPUTS *********************/
	H = NormalizeInputs( H);

	return( H);
}

/***************************************************************************/
/*                                                                         */
/*                               IAC Network                               */
/*                                                                         */
/***************************************************************************/

/* Network constants are taken from McClelland & Rumelhart */
/* PDP Handbook p. 40 */
#define ESTR            0.40    /* scales external (i.e. semantic) input */
#define ALPHA           0.10    /* scales excitory input */
#define GAMMA           0.10    /* scales inhibitory input */
#define MAX             1.00    /* maximum activation */
#define MIN            -0.20    /* minimum activation */
#define DECAY           0.10    /* activation decay rate */
#define REST           -0.10    /* resting activation */
#define MINCHANGE       0.00001 /* minimum change for convergence */

/* The getnet() routine is taken from McClelland & Rumelhart */
/* PDP Handbook pp. 21-23 */
void getnet( H)
struct HYPOTHESIS_NET *H;
{
	int i, j;
	float excitation[MAX_HYPOS];
	float inhibition[MAX_HYPOS];

	for (i=0; i < H->num_hypos; i++) {
		excitation[i] = inhibition[i] = 0.0;

		for (j=0; j < H->num_hypos; j++) {
		    if (H->activation[j] > 0.0) {
		       if (H->w[i][j] > 0) {
			  excitation[i] += H->w[i][j] * H->activation[j];
			  }
		       else if (H->w[i][j] < 0) {
			  inhibition[i] += H->w[i][j] * H->activation[j];
			  }
		       }
		    } /* end for j */

		H->netinput[i] = ESTR * H->seminput[i]
					+ ALPHA * excitation[i]
					+ GAMMA * inhibition[i];
		} /* end for i */
}

/* The update routine is taken from McClelland & Rumelhart */
/* PDP Handbook pp. 21-3 */
int update( H )
struct HYPOTHESIS_NET *H;
{
	int i;
	int converged;
	float change;

	converged = TRUE; /* initialize convergence flag */

	for (i=0; i < H->num_hypos; i++) {
		if (H->netinput[i] > 0.0)
		     change = (MAX - H->activation[i]) * H->netinput[i]
					- DECAY * (H->activation[i] - REST);
		else change = (H->activation[i] - MIN) * H->netinput[i]
					- DECAY * (H->activation[i] - REST);

		/* update convergence flag */
		converged = converged && (change < MINCHANGE);

		/* update activation of unit */
		H->activation[i] += change;

		if (H->activation[i] > MAX) H->activation[i] = MAX;
		if (H->activation[i] < MIN) H->activation[i] = MIN;
		} /* end for i */
	return( converged);
}

#define MAXCYCLES 200 /* maximum number of cycles */
#define COL_START 40
#define FIRST_COL 1
#define ROW_START 2
#define BOTTOM_ROW 20
struct HYPOTHESIS_NET *RunToConvergence( S, H)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
{

	int converged;  /* convergence flag */
	int cycleno;    /* cycle number */
	sequence_ptr index;
	int i;
	cons_ptr x;
	char str[50];

	PrintHypoNames( S, H);
	converged = FALSE;      /* not converged yet */
	cycleno = 0;            /* no cycles yet */
	while (not( converged) && (cycleno < MAXCYCLES)) {
		/* compute the new activations */
		cycleno++;
		getnet( H);
		converged = update( H);
		/* print the new activations */
		_settextposition( ROW_START, COL_START);
		for (i = 0; i < H->num_hypos; i++) {
			sprintf(str, "%+7.4f", H->activation[i]);
			_settextposition( ROW_START+i, COL_START);
			_outtext( str);
			}
		}

	_settextposition( BOTTOM_ROW, FIRST_COL);
	sprintf(str, "Network converged on cycle %d", cycleno);
	_outtext( str);

	return( H);
}


/***************************************************************************/
/*                   Find Winning Match Hypotheses                         */
/***************************************************************************/
int grammar_category_ok( S, target_item)
struct SEMANTIC_NETWORK *S;
int target_item;
{
	if (GetType(S,target_item) == NOUN) return( TRUE);
	else if (GetType(S,target_item) == ADJECTIVE) return( TRUE);
	else if (GetType(S,target_item) == ADVERB) return( TRUE);
	else if (GetType(S,target_item) == SFRAME) return( TRUE);
	else if (GetType(S,target_item) == PROPERTYTYPE) return( TRUE);
	else return( FALSE);
}

sequence_ptr FindWinningMatchHypotheses( S, H, Target, final_winners)
struct SEMANTIC_NETWORK *S;
struct HYPOTHESIS_NET *H;
cons_ptr Target;
sequence_ptr final_winners;
{

	cons_ptr tptr;
	int target_item;
	int second_item;
	int mhpos;
	sequence_ptr mhptr, nhptr;
	cons_ptr x;
	sequence_ptr the_hypos;
	cons_ptr winner;
	float max_activation;
	sequence_ptr the_winners;
	int i;
	sequence_ptr index;
	cons_ptr the_seconds;

	/* initialize the sequence of winning match hypotheses */
	the_winners = NIL;

	/* The first major loop reduces the many-many map to a     */
	/* many-1 by choosing a winner from competing hypotheses   */
	/* of the form                                             */
	/*      (targ_term, src_term1) (targ_term, src_term2)      */
	/* This may still leave the map many-1, for instance, if   */
	/* the map has match hypos like                            */
	/*      (targ_term1, src_term)  (targ_term2, src_term)     */
	/* The second major loop reduces many-1 map to a 1-1 map   */
	/* for each item in Target do */

	/* First major loop: reduce from many-many to many-1       */
/*  printf("Reducing map from many-many to many-1\n");*/
	ListIterator( tptr, Target) {
	  target_item = car( tptr);
	  if (grammar_category_ok( S, target_item)) {
/*      printf("Target item is: %s\n", GetName(S,target_item));*/
		/* find all match hypotheses containing that item */
		the_hypos = NIL;
		SequenceIterator( mhptr, H->match_hypos) {
			x = carseq( mhptr); /* x is a match hypothesis */
			if (first(x) == target_item)
			   the_hypos = unionseq( the_hypos, nconseq(x));
			}
/*      printf("Match hypos containing target item:\n");
		print_sequence_names( the_hypos, S);*/

		if (not(nullseq( the_hypos))) {
			/* find x in the_hypos with max activation */
			max_activation = 0.0;
			SequenceIterator( mhptr, the_hypos) {
				x = carseq( mhptr); /* x is a match hypothesis */
				/* find the position of x in H->match_hypos */
				mhpos = get_positionseq( x, H->match_hypos) - 1;
				if (H->activation[mhpos] > max_activation) {
					winner = x;
					max_activation = H->activation[mhpos];
					}
				}

			/* add the winner to the list of winners */
/*          printf("The winner is: (%s,%s)\n",
				GetName(S,first(winner)),
				GetName(S,second(winner)));*/
			the_winners = appendseq( the_winners, nconseq(winner));
/*          strike_any_key();*/  /* pause for debugging */
			} /* end sequence iterator over the_hypos */
		} /* end if grammar category ok source item */
	  }

	/* the map has been reduced from many-many to 1-many     */
	/* assuming we want the map to be 1-1, we must reduce it */
	/* from 1-many to 1-1                                    */
	/* Second major loop: reduce from many-many to many-1    */
/*  printf("Reducing map from many-1 to 1-1\n");*/

	final_winners = NIL;

	/* make a list of all second terms in the_winners */
	the_seconds = NIL;
	SequenceIterator( mhptr, the_winners) {
	    x = carseq( mhptr); /* x is a match hypothesis */
	    the_seconds = list_union( the_seconds, ncons(second(x)));
	    }

	ListIterator( tptr, the_seconds) {
		second_item = car( tptr);
/*      printf("Second item is: %s\n", GetName(S,second_item));*/
		/* find all match hypotheses containing that item */
		the_hypos = NIL;
		SequenceIterator( mhptr, H->match_hypos) {
			x = carseq( mhptr); /* x is a match hypothesis */
			if (second(x) == second_item)
			   the_hypos = unionseq( the_hypos, nconseq(x));
			}
/*      printf("Match hypos containing second item:\n");
		print_sequence_names( the_hypos, S);*/

		/* find x in the_hypos with max activation */
		max_activation = 0.0;
		SequenceIterator( mhptr, the_hypos) {
			x = carseq( mhptr); /* x is a match hypothesis */
			/* find the position of x in H->match_hypos */
			mhpos = get_positionseq( x, H->match_hypos) - 1;
			if (H->activation[mhpos] > max_activation) {
				winner = x;
				max_activation = H->activation[mhpos];
				}
			}

		/* add the winner to the list of winners */
/*      printf("The winner is: (%s,%s)\n",
			GetName(S,first(winner)),
			GetName(S,second(winner)));*/
		final_winners = appendseq( final_winners, nconseq(winner));
/*      strike_any_key();*/  /* pause for debugging */
		}

	/* print winning match hypotheses */
	printf("----- Winning Match Hypotheses -----   --Activations-- \n");
	i = 0;
	/* for every winning match hypothesis x do */
	SequenceIterator( index, final_winners) {
		x = carseq( index);
		/* find the position of x in H->match_hypos */
		mhpos = get_positionseq( x, H->match_hypos) - 1;
		printf("(%-15.15s = %-15.15s)       %+7.4f\n",
			GetName(S,first(x)),
			GetName(S,second(x)),
			H->activation[mhpos]);
		}
	printf("\n");
	strike_any_key();

	return( final_winners);
}

/* This function makes a map. It is called by the elaboration function.*/
sequence_ptr RunHypothesisNet( S, target, source, TheWinners )
struct SEMANTIC_NETWORK *S;     /* the semantic network */
cons_ptr target;                /* the target field */
cons_ptr source;                /* the source field */
sequence_ptr TheWinners;           /* the winning match hypotheses */
{
	cons_ptr target_verbs;
	int i;
	char TmpStr[MAX_STRLEN];

	struct TextBuffer *FB;      /* for semantic fields */
	struct TextBuffer *AB;      /* for the map generation process */
	struct TextBuffer *CB;      /* for hypothesis net connections */
	struct HYPOTHESIS_NET *H;   /* the hypothesis net */

	/* check the fields */
	if (null( target)) {
		printf("ERROR: Target field is nil.\n");
		exit( ERROR_VALUE);
		}
	if (null( source)) {
		printf("ERROR: Source field is nil.\n");
		exit( ERROR_VALUE);
		}

	/* print the fields */
	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);
	StringMove( FB, "------- Please Scroll Down -------");
	NextLine(FB);
	StringMove( FB, "The target field is: ");
	NextLine(FB);
	FillFieldBuffer( S, FB, nconseq( target));
	StringMove( FB, "The source field is: ");
	NextLine(FB);
	FillFieldBuffer( S, FB, nconseq( source));
	StringMove( FB, "***** End of Text *****");
	NextLine(FB);
	DisplayBuffer( FB, FieldScrollWindow);

	/* find the target verbs */
	target_verbs = NIL;
	target_verbs = select_target_verbs(S, target, target_verbs);
/*  printf("Target verbs: ");
	print_node_names( target_verbs, S); printf("\n"); */


	/*****************************************************************/
	/*                  BUILD THE HYPOTHESIS NETWORK                 */
	/*****************************************************************/
	DisplayWindow( FieldChoiceWindow);
	DisplayExtraMsg( FieldChoiceWindow,
						"Building hypothesis network.....            ");

	/* point at the hypothesis net */
	H = AllocateHypothesisNet();

	/* allocate & initialize text buffer for generating analogies */
	AB = AllocateTextBuffer();
	InitializeTextBuffer( AB);

	/* initialize the hypothesis network */
	StringMove( AB, "Initializing the hypothesis net.");
	NextLine( AB);
	H = InitializeHypothesisNet( H);

	StringMove( AB, "Building the hypothesis net.");
	NextLine( AB);
	H = BuildHypothesisNet( S, H, AB, target_verbs, target, source);
	NextLine( AB);

	CountConnections(H);
	StringMove( AB, "Net contains ");
	itoa(TotalExcitory, TmpStr, 10);
	StringMove( AB, TmpStr);
	StringMove( AB, " EXCITATORY connections.");
	NextLine( AB);
	StringMove( AB, "Net contains ");
	itoa(TotalInhibitory, TmpStr, 10);
	StringMove( AB, TmpStr);
	StringMove( AB, " INHIBITORY connections.");
	NextLine( AB);
	DisplayBuffer( AB, AnalogyScrollWindow);

	if ((TotalExcitory == 0) && (TotalInhibitory == 0)) {
		_clearscreen( _GCLEARSCREEN);
		DisplayWindow( MessageWindow);
		posprint(8, 17, "Network contains no connections!");
		posprint(9, 17, "No analogies will be generated.");
		ExitFromWindow( MessageWindow);
		return( NIL);
		}

	/* Fill the HypoConnBuffer and display it */
	CB = AllocateTextBuffer();
	InitializeTextBuffer( CB);
	FillHypoConnBuffer( S, H, CB);
	BackUpOneLine( CB);
	DisplayBuffer( CB, HypoScrollWindow);

	/* Print the semantic input table */
	PrintSemInputs( S, H);

	/********************************************************************/
	/*            RUN THE HYPOTHESIS NETWORK TO CONVERGENCE             */
	/********************************************************************/
	_clearscreen( _GCLEARSCREEN);
	printf("\n\nRun the hypothesis net to convergence.\n");
	H = RunToConvergence( S, H);
	strike_any_key();   /* pause for observation */

	/* find winning match hypotheses */
	_clearscreen( _GCLEARSCREEN);
	printf("\n\nFind the winning match hypotheses.\n");
	TheWinners = FindWinningMatchHypotheses( S, H, target, TheWinners);

	free(FB); /* free field text buffer */
	free(AB); /* free analogy text buffer */
	free(CB); /* free connections text buffer */
	free(H);  /* free the hypothesis net */

	return( TheWinners);
}

