/***********************************************************************/
/*                                                                     */
/*                           INITIALIZATION FUNCTIONS                  */
/*                Copyright (C) 1994 Eric Charles Steinhart            */
/*                            All rights reserved.                     */
/*                                                                     */
/***********************************************************************/

#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\dos.h"
#include "C:\qc25\include\stdlib.h"
#include "anadefs.h"
#include "semnet.h"
#include "iterate.h"

#define QUERY_ROW   12  /* row for query message in init window */

static char LookAheadBuffer[50];       /* needed to handle FIELD keyword */
static int LookAheadBufferIsLoaded;


/***********************************************************************/
/*                                                                     */
/*                    Predicates in the Input Files                    */
/*                                                                     */
/***********************************************************************/
#define SUPERTYPE_PREDICATE         1   /* "supertype"   */
#define SUBTYPE_PREDICATE           2   /* "subtype"     */
#define STRUCTURAL_PREDICATE        3   /* "contains"    */
#define CONTRASTIVE_PREDICATE       4   /* "opposition"  */
#define GENITIVE_PREDICATE          5   /* "of"          */
#define CONJUNCTION_PREDICATE       6   /* "and"         */
#define RULE_PREDICATE              7   /* "if"          */
#define PROPERTY_PREDICATE          8   /* "property"    */
#define STATE_PREDICATE             9   /* "feature"     */
#define DESCRIPTION_PREDICATE       10  /* "the"         */
#define VERB_PREDICATE              11  /* anything else */

#define MAX_STRUCTVERBS             1
extern char struct_verbs[MAX_STRUCTVERBS][MAX_STRLEN] = {
	"contains"};

#define MAX_CONTRAST_RELS           1
extern char contrastive_rels[MAX_CONTRAST_RELS][MAX_STRLEN] = {
	"opposition"};


/***********************************************************************/
/*                                                                     */
/*                       Network Definition Structure                  */
/*                                                                     */
/***********************************************************************/
/* the this_conn structure is used to hold a connection */
struct THIS_CONN {
	int     from_node;
	int     to_node;
	int     role;
	int     mode;
	int     surface_pos;
	};

struct NetDefStructure {
	int num_nodes;              /* number of nodes */
	int frame_num;              /* used to make frame identifiers */
	char the_nodes[MAX_NODES][MAX_STRLEN];
	char the_types[MAX_NODES][MAX_STRLEN];
	int num_conn;               /* number of connections */
	struct THIS_CONN connections[MAX_CONNECTIONS];
	int InField;                /* TRUE if in explicit semantic field */
	cons_ptr current_field;     /* terms in current explicit field */
								/* NOTE: members of current_field */
								/* are indexes into the_nodes */
	sequence_ptr fields;        /* the explicit fields in the network */
								/* NOTE: members of fields */
								/* are lists of indexes into the_nodes */
	};


/***********************************************************************/
/*                                                                     */
/*                            UTILITY FUNCTIONS                        */
/*                                                                     */
/***********************************************************************/
void ErrorTest( test, message)
int test;
char *message;
{
	if (test) {
		_clearscreen( _GCLEARSCREEN);
		printf("%s\n", message);
		exit( ERROR_VALUE);
		}
}


/* This function looks up a string in a table of strings. */
int name_lookup( str, str_table, str_table_length)
char str[];
char str_table[][MAX_STRLEN];
int str_table_length;
{
	int i;
	for (i=0; i<str_table_length; i++) {
/*              printf("Comparing %s to %s\n", str, str_table[i]);*/
		if (strcmp( str, str_table[i]) == 0) return( i);
		}
	return( STR_NOT_FOUND_ERROR);
}


/***********************************************************************/
/*                                                                     */
/*                    NETWORK DEFINITION STRUCTURE METHODS             */
/*                                                                     */
/***********************************************************************/

#define GetNDefNumNodes( N)     (N->num_nodes)
#define SetNDefNumNodes( N, v)  (N->num_nodes = v)
#define GetNDefNumConn( N)      (N->num_conn)
#define SetNDefNumConn( N, v)   (N->num_conn = v)
#define SetFrameNum( N, v)      (N->frame_num = v)
#define GetTheNodes( N, i)      (N->the_nodes[i])
#define SetTheNodes( N, i, v)   (N->the_nodes[i] = v)
#define GetTheTypes( N, i)      (N->the_types[i])
#define SetTheTypes( N, i, v)   (N->the_types[i] = v)
#define GetCurrentField( N)     (N->current_field)
#define SetCurrentField( N, v)  (N->current_field = v)
#define GetInField( N)          (N->InField)
#define SetInField( N, v)       (N->InField = v)


struct NetDefStructure *AllocateNetDef( void)
{
	struct NetDefStructure *NetDefPtr;

	printf("ENTERING: AllocateNetDef\n");
	printf("About to enter malloc\n");
	NetDefPtr = (struct NetDefStructure *)
					malloc((size_t) sizeof( struct NetDefStructure));
	printf("malloc returned: %d\n", (int)NetDefPtr);
	if (NetDefPtr == (struct NetDefStructure *) NULL) {
		printf("ERROR: Can't allocate network definition structure.\n");
		printf("ERROR: Out of memory.\n");
		exit( ERROR_VALUE);
		}
	printf("EXITING: AllocateNetDef\n");
	return( NetDefPtr);
}


void InitializeNetDef( N)
struct NetDefStructure *N;
{
	N->num_nodes = 0;
	N->num_conn = 0;
	N->frame_num = 0;
	N->fields = NIL;
}


void FreeNetDef( N)
struct NetDefStructure *N;
{
	FreeList( N->current_field);
/*  FreeSequence( N->fields);*/
	free( N);
}


/* prints the members of an explicit semantic field as named nodes */
/* stored in the NetDefStructure */
void print_explicit_field( list, N)
cons_ptr list;
struct NetDefStructure *N;
{
	cons_ptr t;
	t = list;
	if ( null( t))
		printf("NIL\n");
	else while ( not( null( t))) {
		printf( "%s ", GetTheNodes( N, car( t)));
		t = cdr( t);
		}
	printf("\n");
}


void EnterTerm( N, term, gramm_cat)
struct NetDefStructure *N;
char *term;
char *gramm_cat;
{
	int index;
	/* don't add the term if its already in N->the_nodes */
	index = name_lookup( term, N->the_nodes, GetNDefNumNodes( N));
	if (index == STR_NOT_FOUND_ERROR) {
/*      printf("Term %s is not in nodes.\n", term);*/
		strcpy( GetTheNodes( N, GetNDefNumNodes( N)), term);
		strcpy( GetTheTypes( N, GetNDefNumNodes( N)), gramm_cat);
/*      printf("Entered term: %s\n", GetTheNodes( N, GetNDefNumNodes( N)));*/
		GetNDefNumNodes( N)++;
/*      printf("Number of nodes entered: %d\n", GetNDefNumNodes( N));*/
		ErrorTest(
			(GetNDefNumNodes( N) > MAX_NODES),
			"ERROR: Exceeded maximum number of nodes.\n");
		}

	/* if there is a current field, union the term to it */
	if ( GetInField( N)) {
		index = name_lookup( term, N->the_nodes, GetNDefNumNodes( N));
		if (not( is_member( index, GetCurrentField( N))))
			SetCurrentField( N, cons( index, GetCurrentField( N)));
		}
}

int EnterConn( N, from, to, r, m, sp)
struct NetDefStructure *N;  /* pointer to the connections buffer */
char *from;                 /* the from node name */
char *to;                   /* the to node name */
char *r;                    /* the role name */
char *m;                    /* the mode name */
char *sp;                   /* the surface position */
{
	int rindex, cindex, this_role, this_mode, surface_position;

	rindex = name_lookup( from, N->the_nodes, GetNDefNumNodes( N));
	if (rindex == STR_NOT_FOUND_ERROR) {
		printf("ERROR:Unrecognized from node name <%s>.\n", from);
		exit( ERROR_VALUE);
		}

	cindex = name_lookup( to, N->the_nodes, GetNDefNumNodes( N));
	if (cindex == STR_NOT_FOUND_ERROR) {
		printf("ERROR:Unrecognized to node name %s.\n", to);
		exit( ERROR_VALUE);
		}

	this_role = name_lookup(r, role_names, MAX_ROLES);
	if (this_role == STR_NOT_FOUND_ERROR) {
		printf("ERROR:Unrecognized role name %s.\n", r);
		exit( ERROR_VALUE);
		}

	this_mode = name_lookup(m, mode_names, MAX_MODES);
	if (this_mode == STR_NOT_FOUND_ERROR) {
		printf("ERROR:Unrecognized mode name %s.\n", m);
		exit( ERROR_VALUE);
		}

	sscanf(sp, "%d", &surface_position);

	/* emit a connection into the connections buffer */
	N->connections[GetNDefNumConn( N)].from_node = rindex;
	N->connections[GetNDefNumConn( N)].to_node = cindex;
	N->connections[GetNDefNumConn( N)].role = this_role;
	N->connections[GetNDefNumConn( N)].mode = this_mode;
	N->connections[GetNDefNumConn( N)].surface_pos = surface_position;

	/* increment number of connections written */
	GetNDefNumConn( N)++;
	ErrorTest(
		(GetNDefNumConn( N) >= MAX_CONNECTIONS),
		"ERROR:Proposition file generated too many connections.\n");
}

int VerifyNetMembership( N, term)
struct NetDefStructure *N;
char *term;
{
	int index;
	index = name_lookup(term, N->the_nodes, GetNDefNumNodes( N));
	if (index == STR_NOT_FOUND_ERROR) {
		_clearscreen( _GCLEARSCREEN);
		printf(
		   "ERROR:Conjunction or rule argument <%s> is not defined.\n",
		   term);
		exit(ERROR_VALUE);
		}
	return( TRUE);
}

int VerifyTypeIsFrame(N, term)
struct NetDefStructure *N;
char *term;
{
	int index;

	index = name_lookup(term, N->the_nodes, GetNDefNumNodes( N));
	if (strcmp( GetTheTypes( N, index), "SFRAME") == 0)
		return( TRUE);
	else if (strcmp( GetTheTypes( N, index), "MFRAME") == 0)
		return( TRUE);
	else if (strcmp( GetTheTypes( N, index), "CONJUNCTION") == 0)
		return( TRUE);
	else {
		_clearscreen( _GCLEARSCREEN);
		printf(
		  "ERROR:Conjunction or rule argument <%s> is not a proposition.\n",
		  term);
		exit(ERROR_VALUE);
		}
}


/***********************************************************************/
/*                                                                     */
/*                           TOKENIZER FUNCTIONS                       */
/*                                                                     */
/***********************************************************************/
#define LETTER  'a'
#define DIGIT   '0'
int TypeOfChar(ch)
char ch;
{
	if ((ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z'))
		return( LETTER);
	else if (ch >= '1' && ch <= '9')
		return( DIGIT);
	else return( ch);
}

char GetNextChar(fp)
FILE *fp;
{
	char ch;
	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);
	return(ch);  /* returns first non-white space char */
}

void GetColon(fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* check for colon */
	if (ch != ':') {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:Got <%c> but expected colon.\n", ch);
		exit( ERROR_VALUE);
		}
}

void GetDash(fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* check for dash */
	if (ch != '-') {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR:Got <%c> but expected dash.\n", ch);
		exit( ERROR_VALUE);
		}
}

void GetOpenBrace(fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* check for open brace */
	if (ch != '{') {
		printf("ERROR:Got <%c> but expected open brace.\n", ch);
		exit( ERROR_VALUE);
		}
}

void GetOpenParen(fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* check for open paren */
	if (ch != '(') {
		printf("ERROR:Got <%c> but expected open parenthesis.\n", ch);
		exit( ERROR_VALUE);
		}
}

void GetCloseParen(fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* check for close paren */
	if (ch != ')') {
		printf("ERROR:Got <%c> but expected close parenthesis.\n", ch);
		exit( ERROR_VALUE);
		}
}

void GetComma(fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* check for comma */
	if (ch != ',') {
		printf("ERROR:Got <%c> but expected comma.\n", ch);
		exit( ERROR_VALUE);
		}
}

int GetCloseParenOrComma( fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* read the next punctuation mark */
	if (ch == ',')
		return( TRUE); /* process more arguments */
	else if (ch == ')')
		return( FALSE); /* end of proposition */
	else {
		printf("ERROR:Expected comma or close paren.\n");
		exit( ERROR_VALUE);
		}
}

int GetParenOrComma( fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* read the next punctuation mark */
	if (ch == ',')
		return( COMMA);
	else if (ch == '(')
		return( OPEN_PAREN);
	else if (ch == ')')
		return( CLOSE_PAREN);
	else {
		printf("ERROR:Expected comma or paren.\n");
		exit( ERROR_VALUE);
		}
}

int GetValueListPunct( fp)
FILE *fp;
{
	char ch;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* read the next punctuation mark */
	if (ch == ',')
		return( TRUE); /* process more values */
	else if (ch == '}')
		return( FALSE); /* end of value list */
	else {
		printf("ERROR:Expected comma or close brace.\n");
		exit( ERROR_VALUE);
		}
}

void GetIdentifier( fp, token )
FILE *fp;
char *token;
{
	char ch;
	int i;

	if (LookAheadBufferIsLoaded) {
/*      printf("Reading from LookAheadBuffer.\n");
		printf("LookAheadBuffer = <%s>", LookAheadBuffer);*/
		strcpy(token, LookAheadBuffer);
		LookAheadBufferIsLoaded = FALSE;
		return;
		}

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	i = 0;
	/* identifier may have leading % */
	if (ch == '%') {
		token[i] = ch;
		ch = getc(fp);
		i++;
		}

	/* read a sequence of alphanumerics */
	/* stop if a non-alphanumeric is encountered */
	while ((TypeOfChar(ch) == LETTER) || (TypeOfChar(ch) == DIGIT)) {
		token[i] = ch;
		ch = getc(fp);
		i++;
		}
	ungetc(ch,fp);
	token[i] = 0;
}

/* reads a role; as a side-effect, fills the_role with the role */
void GetArgumentRole( fp, the_role, role_names)
FILE *fp;
char *the_role;
char role_names[][MAX_STRLEN];
{
	int index;
	/* read the role */
	GetIdentifier( fp, the_role);
/*  printf("The role is: %s\n", the_role);*/
	/* look up the role in the list of roles */
	index = name_lookup(the_role, role_names, MAX_ROLES);
	if (index == STR_NOT_FOUND_ERROR) {
		printf("ERROR:Unrecognized role name <%s>.\n", the_role);
		exit( ERROR_VALUE);
		}
}

void GetTerm( fp, term)
FILE *fp;
char *term;
{
	char LongString[50];
	int MoreToCopy, i;

	GetIdentifier( fp, LongString);
	i = 0;
	MoreToCopy = TRUE;
	while ( MoreToCopy ) {
		term[i] = LongString[i];
		if (term[i] == 0) MoreToCopy = FALSE;
		if (i == (MAX_STRLEN - 1)) {
			term[i] = 0; /* truncate string */
			MoreToCopy = FALSE;
			}
		i++;
		}
/*  printf("The term is: %s\n", term);*/
}

void GetVerb( fp, v)
FILE *fp;
char *v;
{
	GetTerm( fp, v); /* read the verb */
}


int GetVerbType( v)
char *v;
{
	int index;

	/* Check for SUPERTYPE predicate */
	if (strcmp(v, "supertype") == 0)
		   return( SUPERTYPE_PREDICATE);

	/* Check for SUBTYPE predicate */
	if (strcmp(v, "subtype") == 0)
		   return( SUBTYPE_PREDICATE);

	/* Check for STRUCTURAL predicate */
	index = name_lookup( v, struct_verbs, MAX_STRUCTVERBS);
	if (index != STR_NOT_FOUND_ERROR)
		   return( STRUCTURAL_PREDICATE);

	/* Check for CONTRASTIVE predicate */
	index = name_lookup( v, contrastive_rels, MAX_CONTRAST_RELS);
	if (index != STR_NOT_FOUND_ERROR)
			return( CONTRASTIVE_PREDICATE);

	/* Check for GENITIVE predicate */
	if (strcmp(v, "of") == 0)
			return( GENITIVE_PREDICATE);

	/* Check for CONJUNCTION predicate */
	if (strcmp(v, "and") == 0)
			return( CONJUNCTION_PREDICATE);

	/* Check for RULE predicate */
	if (strcmp(v, "if") == 0)
			return( RULE_PREDICATE);

	/* Check for PROPERTY predicate */
	if (strcmp(v, "property") == 0)
			return( PROPERTY_PREDICATE);

	/* Check for STATE predicate */
	if (strcmp(v, "feature") == 0)
			return( STATE_PREDICATE);

	/* Check for DESCRIPTION predicate */
	if (strcmp(v, "the") == 0)
			return( DESCRIPTION_PREDICATE);

	return( VERB_PREDICATE);
}

#define NOUN_ROLE       0
#define ADJECTIVE_ROLE  1
int GetRoleType( r)
char *r;
{
	int index;

	/* lookup role in table of role_names */
	index = name_lookup( r, role_names, MAX_ROLES);
	if (index == STR_NOT_FOUND_ERROR) {
		printf("ERROR:Unrecognized role name.\n");
		exit( ERROR_VALUE);
		}
	else if ((0 <= index) && (index < MOD_ROLES))
		return( NOUN_ROLE );
	else if ((MOD_ROLES <= index) && (index <= MAX_FUNC_ROLES))
		return( ADJECTIVE_ROLE);
	else {
		printf("ERROR:Illegal role type <%s> for verb.\n", r);
		exit( ERROR_VALUE);
		}
}

int GetPositionModified( r)
char *r;
{
	int index;

	/* lookup role in table of role_names */
	index = name_lookup( r, role_names, MAX_ROLES);
	if (index == STR_NOT_FOUND_ERROR) {
		printf("ERROR:Unrecognized role name.\n");
		exit( ERROR_VALUE);
		}
	return( index - MOD_ROLES);
}


/* look for the FIELD keyword */
int GetField( fp)
FILE *fp;
{
	char ch;
	int i;

	/* skip white space */
	ch = getc(fp);
	while ((ch == ' ') || (ch == '\n') || (ch == '\t')) ch = getc(fp);

	/* read a sequence of alphanumerics */
	/* stop if a non-alphanumeric is encountered */
	i = 0;
	while ((TypeOfChar(ch) == LETTER) || (TypeOfChar(ch) == DIGIT)) {
		LookAheadBuffer[i] = ch;
		ch = getc(fp);
		i++;
		}
	ungetc(ch,fp);
	LookAheadBuffer[i] = 0;

/*  printf("In GetField: LookAheadBuffer = <%s>\n", LookAheadBuffer);*/
	if (strcmp(LookAheadBuffer, "FIELD") == 0) {
		LookAheadBufferIsLoaded = FALSE;
		return( TRUE);
		}
	else {
		LookAheadBufferIsLoaded = TRUE;
		return( FALSE); /* keyword FIELD was not encountered */
		}
}


/***********************************************************************/
/*                                                                     */
/*                          PROCESS A PROPOSITION                      */
/*                                                                     */
/***********************************************************************/
SupertypeProposition( fp, N)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
{
	char first_arg[MAX_STRLEN];
	char arg[MAX_STRLEN];
	int StillMoreArguments;

	/* supertype( <genus>, {<species1>, . . . <speciesN>}) */
	GetOpenParen(fp);
	/* process the genus */
	GetTerm(fp, first_arg);
	EnterTerm(N, first_arg, "NOUN");
	GetComma(fp);
	GetOpenBrace(fp);
	StillMoreArguments = TRUE;
	while (StillMoreArguments) {
		GetTerm(fp, arg); /* get a species */
		EnterTerm(N, arg, "NOUN");
		/****************************************************/
		/*  <genus>                                         */
		/*     |-----------SUPERTYPE----------> <species1>  */
		/*     |                                            */
		/*     |-----------SUPERTYPE----------> <species2>  */
		/*     |                                            */
		/*     |                                            */
		/*     |-----------SUPERTYPE----------> <speciesN>  */
		/*                                                  */
		/*  <genus>                                         */
		/*     |<-----------SUBTYPE------------ <species1>  */
		/*     |                                            */
		/*     |<-----------SUBTYPE------------ <species2>  */
		/*     |                                            */
		/*     |                                            */
		/*     |<-----------SUBTYPE------------ <speciesN>  */
		/****************************************************/
		/* emit connection from <genus> to <species> */
		EnterConn(N, first_arg, arg, "SUPERTYPE", "LITERAL", "0");
		/* emit connection from <species> to <genus> */
		EnterConn(N, arg, first_arg, "SUBTYPE", "LITERAL", "0");
		StillMoreArguments = GetValueListPunct(fp);
		}
	GetCloseParen(fp);
}

SubtypeProposition( fp, N)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
{
	char first_arg[MAX_STRLEN];
	char arg[MAX_STRLEN];
	int StillMoreArguments;

	/* subtype( <species>, {<genus1>, . . . <genusN>}) */
	GetOpenParen(fp);
	/* process the species */
	GetTerm(fp, first_arg);
	EnterTerm(N, first_arg, "NOUN");
	GetComma(fp);
	GetOpenBrace(fp);
	StillMoreArguments = TRUE;
	while (StillMoreArguments) {
		GetTerm(fp, arg); /* get a genus */
		EnterTerm(N, arg, "NOUN");
		/* emit connection from <species> to <genus> */
		EnterConn(N, first_arg, arg, "SUBTYPE", "LITERAL", "0");
		/* emit connection from <genus> to <species> */
		EnterConn(N, arg, first_arg, "SUPERTYPE", "LITERAL", "0");
		StillMoreArguments = GetValueListPunct(fp);
		}
	GetCloseParen(fp);
}


StructuralProposition( fp, N)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
{
	char first_arg[MAX_STRLEN];
	char arg[MAX_STRLEN];
	int StillMoreArguments;

	/* contains( <whole>, {<part1>, . . . <partN>}) */
	GetOpenParen(fp);
	/* process the whole */
	GetTerm(fp, first_arg);
	EnterTerm(N, first_arg, "NOUN");
	GetComma(fp);
	GetOpenBrace(fp);
	StillMoreArguments = TRUE;
	while (StillMoreArguments) {
		GetTerm(fp, arg); /* get a part */
		EnterTerm(N, arg, "NOUN");
		/****************************************************/
		/*  <whole>                                         */
		/*     |-------------WHOLE------------> <part1>     */
		/*     |                                            */
		/*     |-------------WHOLE------------> <part2>     */
		/*     |                                            */
		/*     |                                            */
		/*     |-------------WHOLE------------> <partN>     */
		/*                                                  */
		/*  <whole>                                         */
		/*     |<-------------PART------------- <part1>     */
		/*     |                                            */
		/*     |<-------------PART------------- <part2>     */
		/*     |                                            */
		/*     |                                            */
		/*     |<-------------PART------------- <partN>     */
		/****************************************************/
		/* emit connection from first_arg to arg */
		EnterConn(N, first_arg, arg, "WHOLE", "LITERAL", "0");
		/* emit connection from arg to first_arg */
		EnterConn(N, arg, first_arg, "PART", "LITERAL", "0");
		StillMoreArguments = GetValueListPunct(fp);
		}
	GetCloseParen(fp);
}


ContrastiveProposition( fp, N, PropName, v)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
char *PropName;
char *v;
{
	char first_arg[MAX_STRLEN];
	char arg[MAX_STRLEN];
	char rstr[MAX_STRLEN];
	int StillMoreArguments;

	EnterTerm(N, PropName, "PFRAME");
	EnterTerm(N, v, "CONTRASTTYPE");
	/* write connections between contrastive relation */
	/* and frame identifier */
	EnterConn(N, v, PropName, "CONTRAST", "LITERAL", "0");
	EnterConn(N, PropName, v, "CONTRAST", "LITERAL", "0");
	GetOpenParen(fp);
	StillMoreArguments = TRUE;
	while (StillMoreArguments) {
		/* process arguments of paradigmatic proposition */
		GetArgumentRole(fp, rstr, role_names);
		GetColon(fp);
		GetTerm(fp, arg);
		EnterTerm(N, arg, "ADJECTIVE");
		/* write connections between arg and frame id */
		EnterConn(N, arg, PropName, rstr, "LITERAL", "0");
		EnterConn(N, PropName, arg, rstr, "LITERAL", "0");
		StillMoreArguments = GetCloseParenOrComma(fp);
		} /* end while loop that processes arguments */
}


GenitiveProposition( fp, N)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
{
	char first_arg[MAX_STRLEN];
	char second_arg[MAX_STRLEN];
	char arg[MAX_STRLEN];
	int StillMoreArguments;

	GetOpenParen(fp);
	/* process the first argument */
	GetTerm(fp, first_arg);
	EnterTerm(N, first_arg, "NOUN");
	GetComma(fp);
	/* process the second argument */
	GetTerm(fp, second_arg);
	EnterTerm(N, second_arg, "NOUN");
	/* the predicate is the genitive "of"           */
	/* the connection for OF(A, B)   */
	/* is A --- OF ---> B                      */
	EnterConn(N, first_arg, second_arg, "OF", "LITERAL", "0");
	GetCloseParen(fp);
}


ConjunctionProposition( fp, N, PropName)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
char *PropName;
{
	char arg[MAX_STRLEN];
	int StillMoreArguments;

	EnterTerm(N, PropName, "CONJUNCTION");
	GetOpenParen(fp);
	StillMoreArguments = TRUE;
	while (StillMoreArguments) {
		/* process arguments of a conjunction */
		GetTerm(fp, arg);
		VerifyNetMembership(N, arg);
		VerifyTypeIsFrame(N, arg);
		/* the predicate is the conjunction "and"       */
		/* the connection for T:AND(Q1, Q2, . . . Qn)   */
		/* is           T                               */
		/*              |------ANDBRANCH-----Q1         */
		/*              |------ANDBRANCH-----Q2         */
		/*              |                               */
		/*              |------ANDBRANCH-----Qn         */
		/* write connection from arg to frame id */
		EnterConn(N, arg, PropName, "ANDBRANCH", "LITERAL", "0");
		/* write connection from frame id to arg */
		EnterConn(N, PropName, arg, "ANDBRANCH", "LITERAL", "0");
		StillMoreArguments = GetCloseParenOrComma(fp);
		} /* end while loop that processes arguments */
}


RuleProposition( fp, N, PropName)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
char *PropName;
{
	char arg[MAX_STRLEN];
	char rstr[MAX_STRLEN];
	char nstr[MAX_STRLEN];
	char first_arg[MAX_STRLEN];
	char second_arg[MAX_STRLEN];
	char ConjName[MAX_STRLEN];
	cons_ptr ptr, ConseqList;
	int index;
	int StillMoreArguments;

	/*************************************************************/
	/* Rules have the form:                                      */
	/*                                                           */
	/* <PropositionName>:                                        */
	/*        if <Antecedent> then { <Conseq1>, . . . <ConseqN>} */
	/*                                                           */
	/*  The consequent list can have 1 or more members.          */
	/*  if there is only one consequent,                         */
	/*  Antecedent <------> PropositionName <------> Conseq1     */
	/*  if there are many conseqents,                            */
	/*  Antecedent <------> PropositionName                      */
	/*                          |                                */
	/*                      ConjunctionNode                      */
	/*                          |--------------> Conseq1         */
	/*                          |--------------> Conseq2         */
	/*                          |                                */
	/*                          |--------------> ConseqN         */
	/*************************************************************/
	EnterTerm(N, PropName, "RULE");
	/* process the antecedent */
	GetTerm(fp, first_arg);
	VerifyNetMembership(N, first_arg);
	VerifyTypeIsFrame(N, first_arg);
	GetIdentifier(fp, rstr);
	ErrorTest(
		(strcmp(rstr, "then") != 0),
		"ERROR:Rule <%s> does not have a 'then' after antecedent.");
	/* process the consequent(s) */
	GetOpenBrace(fp);
	GetTerm(fp, second_arg);
	VerifyNetMembership(N, second_arg);
	VerifyTypeIsFrame(N, second_arg);
	index = name_lookup(second_arg, N->the_nodes, GetNDefNumNodes( N));
	ConseqList = NIL;
	ConseqList = cons(index, ConseqList);
	StillMoreArguments = GetValueListPunct(fp);
	if (not(StillMoreArguments)) {
		/* connections for T:if P then Q are       */
		/* P----ANTECEDENT----T-----CONSEQUENT----Q */
		/* emit connections between first_arg and PropName */
		EnterConn(N, first_arg, PropName,
			"ANTECEDENT", "LITERAL", "0");
		EnterConn(N, PropName, first_arg,
			"ANTECEDENT", "LITERAL", "0");
		/* emit connections between PropName and second_arg */
		EnterConn(N, PropName, second_arg,
			"CONSEQUENT", "LITERAL", "0");
		EnterConn(N, second_arg, PropName,
			"CONSEQUENT", "LITERAL", "0");
		}
	else { /* consequent is a conjunction of frames */
		while (StillMoreArguments) {
			GetTerm(fp, second_arg);
			VerifyNetMembership(N, second_arg);
			VerifyTypeIsFrame(N, second_arg);
			index = name_lookup(second_arg, N->the_nodes, GetNDefNumNodes( N));
			ConseqList = cons(index, ConseqList);
			StillMoreArguments = GetValueListPunct(fp);
			}
		/* emit connections between first_arg and PropName */
		EnterConn(N, first_arg, PropName, "ANTECEDENT", "LITERAL", "0");
		EnterConn(N, PropName, first_arg, "ANTECEDENT", "LITERAL", "0");

		/* generate a name for the Conjunction */
		strcpy(ConjName, "CONJ");
		itoa(GetNDefNumNodes( N) + 1, nstr, 10);
		strcat(ConjName, nstr);
		EnterTerm(N, ConjName, "CONJUNCTION");

		/* emit connections between PropName and ConjName */
		EnterConn(N, PropName, ConjName, "CONSEQUENT", "LITERAL", "0");
		EnterConn(N, ConjName, PropName, "CONSEQUENT", "LITERAL", "0");

		ListIterator( ptr, ConseqList) {
			index = car(ptr);
			/* write connection from arg to frame id */
			EnterConn(N, GetTheNodes( N, index), ConjName,
				"ANDBRANCH", "LITERAL", "0");
			/* write connection from frame id to arg */
			EnterConn(N, ConjName, GetTheNodes( N, index),
				"ANDBRANCH", "LITERAL", "0");
			}
		}
	FreeList( ConseqList);
}


PropertyProposition( fp, N, PropName)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
char *PropName;
{
	char arg[MAX_STRLEN];
	char rstr[MAX_STRLEN];
	char nstr[MAX_STRLEN];
	char first_arg[MAX_STRLEN];
	char second_arg[MAX_STRLEN];
	char PropertyName[MAX_STRLEN];
	char ObjectName[MAX_STRLEN];
	char Value[MAX_STRLEN];
	char LastValue[MAX_STRLEN];
	int index;
	int StillMoreArguments;

	/* PROPERTY(<propertyname>-of(<object>), IS:{<value1>, . . .}) */
	EnterTerm(N, PropName, "PROPERTYTYPE");
	GetOpenParen(fp);
	/* process the property name */
	GetTerm(fp, PropertyName);
	EnterTerm(N, PropertyName, "NOUN");
	/* enter connections between PropName & PropertyName */
	/* <proposition-name> -----ATTRIBUTE----> <property-name> */
	/* <proposition-name> <----ATTRIBUTE----- <property-name> */
	EnterConn(N,PropertyName, PropName, "ATTRIBUTE", "LITERAL", "0");
	EnterConn(N,PropName, PropertyName, "ATTRIBUTE", "LITERAL", "0");
	GetDash(fp);
	GetIdentifier(fp, rstr);
	ErrorTest(
		(strcmp(rstr, "of") != 0),
		"ERROR:Property <%s> does not have an 'of' after dash.");
	GetOpenParen(fp);
	GetIdentifier(fp, ObjectName);
	EnterTerm(N, ObjectName, "NOUN");
	/* enter connections between ObjectName & PropName */
	/* <object-name> -----PROPERTY----> <proposition-name> */
	/* <object-name> <-----OBJECT------ <proposition-name> */
	EnterConn(N, ObjectName, PropName, "PROPERTY", "LITERAL", "0");
	EnterConn(N, PropName, ObjectName, "OBJECT", "LITERAL", "0");
	GetCloseParen(fp);
	GetComma(fp);
	GetIdentifier(fp, rstr);
	ErrorTest(
		(strcmp(rstr, "is") != 0),
		"ERROR:Property <%s> does not have an 'is' after comma.");
	GetColon(fp);
	GetOpenBrace(fp);
	StillMoreArguments = TRUE;
	LastValue[0] = NIL;
	while (StillMoreArguments) {
		/* process values of a property */
		GetTerm(fp, Value);
		EnterTerm(N, Value, "ADJECTIVE");
		/* enter connections between Value & PropertyName */
		/* <value> -----SUPERTYPE----> <property-name> */
		/* <value> <-----SUBTYPE------ <property-name> */
		EnterConn(N, Value, PropertyName, "SUBTYPE", "LITERAL", "0");
		EnterConn(N, PropertyName, Value, "SUPERTYPE", "LITERAL", "0");
		/* enter connections between Value & PropName */
		/* <value> -----VALUE----> <proposition-name> */
		/* <value> <----VALUE----- <proposition-name> */
		EnterConn(N, Value, PropName, "VALUE", "LITERAL", "0");
		EnterConn(N, PropName, Value, "VALUE", "LITERAL", "0");
		/* add SUCCESSOR and PREDECESSOR links between values */
		if (not(LastValue[0] == NIL)) {
			EnterConn(N, LastValue, Value, "SUCCESSOR", "LITERAL", "0");
			EnterConn(N, Value, LastValue, "PREDECESSOR", "LITERAL", "0");
			}
		strcpy( LastValue, Value);
		StillMoreArguments = GetValueListPunct(fp);
		} /* end while loop that processes values */
	GetCloseParen(fp);
}


FeatureProposition( fp, N, PropName)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
char *PropName;
{
	char arg[MAX_STRLEN];
	char rstr[MAX_STRLEN];
	char PropertyName[MAX_STRLEN];
	char ObjectName[MAX_STRLEN];
	char Value[MAX_STRLEN];
	char LastValue[MAX_STRLEN];
	int StillMoreArguments;

	/* feature(<propertyname>-of(<object>), is:<value>) */
	EnterTerm(N, PropName, "MFRAME");
	GetOpenParen(fp);
	GetTerm(fp, PropertyName);
	EnterTerm(N, PropertyName, "NOUN");
	GetDash(fp);
	GetIdentifier(fp, rstr);
	ErrorTest(
		(strcmp(rstr, "of") != 0),
		"ERROR:Property <%s> does not have an 'of' after dash.");
	GetOpenParen(fp);
	GetIdentifier(fp, ObjectName);
	EnterTerm(N, ObjectName, "NOUN");
	GetCloseParen(fp);
	GetComma(fp);
	GetIdentifier(fp, rstr);
	ErrorTest(
		(strcmp(rstr, "is") != 0),
		"ERROR:Property <%s> does not have an 'is' after comma.");
	GetColon(fp);
	GetIdentifier(fp, Value);
	EnterTerm(N, Value, "ADJECTIVE");
	GetCloseParen(fp);
	/* connections for feature are */
	/*     <ObjectName> <---SUBJECT---> <PropName> */
	/*     <PropName> has <ObjectName> as a SUBJECT */
	/*     <PropName> <----PREDICATE----> <Value> */
	/*     <PropName> has <Value> as a PREDICATE */
	/* write connections between ObjectName and Value */
	EnterConn(N, PropName,   ObjectName, "SUBJECT",   "LITERAL", "0");
	EnterConn(N, ObjectName, PropName,   "SUBJECT",   "LITERAL", "0");
	EnterConn(N, PropName,   Value,      "PREDICATE", "LITERAL", "0");
	EnterConn(N, Value,      PropName,   "PREDICATE", "LITERAL", "0");
}


DescriptionProposition( fp, N, PropName)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
char *PropName;
{
	char arg[MAX_STRLEN];
	char rstr[MAX_STRLEN];
	int StillMoreArguments;
	int SurfacePosition;
	char SurfPos[MAX_STRLEN];

	EnterTerm(N, PropName, "DESCRIPTION");
	GetOpenParen(fp);

	/* process first noun; it's never preceded by a preposition */
	SurfacePosition = 1;
	GetTerm(fp, arg);
	EnterTerm(N, arg, "NOUN");
	/* write connections between arg and frame id */
	itoa(SurfacePosition, SurfPos, 10);
	EnterConn(N, arg, PropName, "AGENT", "LITERAL", SurfPos);
	EnterConn(N, PropName, arg, "AGENT", "LITERAL", SurfPos);

	StillMoreArguments = GetCloseParenOrComma(fp);
	if (not(StillMoreArguments)) {
		printf("ERROR: Descriptions require more than one noun.\n");
		exit( ERROR_VALUE);
		}
	while (StillMoreArguments) {
		/* process remaining arguments of generic description */
		SurfacePosition++;
		GetArgumentRole(fp, rstr, role_names);
		GetColon(fp);
		GetTerm(fp, arg);
		EnterTerm(N, arg, "NOUN");
		/* write connections between arg and frame id */
		itoa(SurfacePosition, SurfPos, 10);
		EnterConn(N, arg, PropName, rstr, "LITERAL", SurfPos);
		EnterConn(N, PropName, arg, rstr, "LITERAL", SurfPos);
		StillMoreArguments = GetCloseParenOrComma(fp);
		} /* end while loop that processes arguments */
}


VerbProposition( fp, N, PropName, v)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
char *PropName;
char *v;
{
	/* string buffers */
	char arg[MAX_STRLEN];
	char rstr[MAX_STRLEN];
	char nstr[MAX_STRLEN];
	int RoleNum;
	int Punct;
	cons_ptr RolesEncountered;
	int StillMoreArguments;
	int SurfacePosition;
	char SurfPos[MAX_STRLEN];

	EnterTerm(N, PropName, "SFRAME");
	EnterTerm(N, v, "VERBTYPE");
	/* write connections between verb and frame identifier */
	EnterConn(N, v, PropName, "VERB", "LITERAL", "2");
	EnterConn(N, PropName, v, "VERB", "LITERAL", "2");

	GetOpenParen(fp);
	StillMoreArguments = TRUE;
	SurfacePosition = 1;
	RolesEncountered = NIL;
	while (StillMoreArguments) {
		/* process arguments of functional proposition */
		GetArgumentRole(fp, rstr, role_names);
		/* check for duplicate role */
		RoleNum = name_lookup(rstr, role_names, MAX_ROLES);
		if (is_member(RoleNum, RolesEncountered)) {
			_clearscreen( _GCLEARSCREEN);
			printf("ERROR: Role <%s> occurs more than once\n",
					role_names[RoleNum]);
			printf("       in a proposition with verb <%s>.\n", v);
			exit(ERROR_VALUE);
			}
		else RolesEncountered = cons(RoleNum, RolesEncountered);
		GetColon(fp);
		GetTerm(fp, arg);
		Punct = GetParenOrComma(fp);
		switch (Punct) {
			case OPEN_PAREN:
				EnterTerm(N, arg, "ADJECTIVE");
				/* write connection from noun arg to frame id */
				itoa(SurfacePosition, SurfPos, 10);
				strcpy(nstr, role_names[RoleNum + MOD_ROLES]);
				EnterConn(N, arg, PropName, nstr, "LITERAL", SurfPos);
				/* write connection from frame id to noun arg */
				EnterConn(N, PropName, arg, nstr, "LITERAL", SurfPos);
				GetTerm(fp, arg);
				EnterTerm(N, arg, "NOUN");
				/* write connection from noun arg to frame id */
				itoa(SurfacePosition, SurfPos, 10);
				EnterConn(N, arg, PropName, rstr, "LITERAL", SurfPos);
				/* write connection from frame id to noun arg */
				EnterConn(N, PropName, arg, rstr, "LITERAL", SurfPos);
				if (SurfacePosition == 1)
					SurfacePosition = 3; /* must skip verb position */
				else SurfacePosition++;
				GetCloseParen(fp);
				StillMoreArguments = GetCloseParenOrComma(fp);
				break;
			case COMMA:
				EnterTerm(N, arg, "NOUN");
				/* write connection from noun arg to frame id */
				itoa(SurfacePosition, SurfPos, 10);
				EnterConn(N, arg, PropName, rstr, "LITERAL", SurfPos);
				/* write connection from frame id to noun arg */
				EnterConn(N, PropName, arg, rstr, "LITERAL", SurfPos);
				if (SurfacePosition == 1)
					SurfacePosition = 3; /* must skip verb position */
				else SurfacePosition++;
				StillMoreArguments = TRUE;
				break;
			case CLOSE_PAREN:
				EnterTerm(N, arg, "NOUN");
				/* write connection from noun arg to frame id */
				itoa(SurfacePosition, SurfPos, 10);
				EnterConn(N, arg, PropName, rstr, "LITERAL", SurfPos);
				/* write connection from frame id to noun arg */
				EnterConn(N, PropName, arg, rstr, "LITERAL", SurfPos);
				if (SurfacePosition == 1)
					SurfacePosition = 3; /* must skip verb position */
				else SurfacePosition++;
				StillMoreArguments = FALSE;
				break;
			default:
				printf("ERROR: Expected paren or comma.\n");
				exit(ERROR_VALUE);
				break;
			}
		} /* end while loop that processes arguments */
	FreeList( RolesEncountered);
}


void ProcessProposition( fp, N)
FILE *fp;                   /* pointer to proposition file  */
struct NetDefStructure *N;  /* pointer to net def structure */
{
	char v[MAX_STRLEN];
	char PropName[MAX_STRLEN];
	int VerbType;   /* type of the verb */

	/* Get the proposition's name */
	GetTerm(fp, PropName);
	GetColon(fp);

	/* Get the proposition's predicate */
	GetVerb(fp, v);
	VerbType = GetVerbType( v);

	switch (VerbType) {
		case SUPERTYPE_PREDICATE:
			SupertypeProposition( fp, N);
			break;
		case SUBTYPE_PREDICATE:
			SubtypeProposition( fp, N);
			break;
		case STRUCTURAL_PREDICATE:
			StructuralProposition( fp, N);
			break;
		case CONTRASTIVE_PREDICATE:
			ContrastiveProposition( fp, N, PropName, v);
			break;
		case GENITIVE_PREDICATE:
			GenitiveProposition( fp, N);
			break;
		case CONJUNCTION_PREDICATE:
			ConjunctionProposition( fp, N);
			break;
		case RULE_PREDICATE:
			RuleProposition( fp, N, PropName);
			break;
		case PROPERTY_PREDICATE:
			PropertyProposition( fp, N, PropName);
			break;
		case STATE_PREDICATE:
			FeatureProposition( fp, N, PropName);
			break;
		case DESCRIPTION_PREDICATE:
			DescriptionProposition( fp, N, PropName);
			break;
		case VERB_PREDICATE:
			VerbProposition( fp, N, PropName, v);
			break;
		default:
			_clearscreen( _GCLEARSCREEN);
			printf("ERROR:Unknown predicate type for <%s>\n", v);
			exit( ERROR_VALUE);
		   }

		/* print the contents of N->the_nodes */
/*      for (index = 0; index < GetNDefNumNodes( N); index++)
			printf("N->the_nodes[%d] = %s\n",
						index, GetTheNodes( N, index));*/
}


/* PropFileToNetDefFile() transforms a list of propositions */
/* into a network definition structure */
void PropFileToNetDefFile( fp, prop_name, N)
FILE *fp;                   /* pointer to proposition file  */
char *prop_name;            /* name of proposition file */
struct NetDefStructure *N;
{

	int PropsInFile;    /* number of propositions in proposition file */
	int index;
	int FoundCloseBrace;
	char NextChar;

	/* read the number of propositions. */
/*  printf("Reading the number of propositions.\n"); */
	fscanf( fp, "%d", &PropsInFile);

	while (N->frame_num < PropsInFile) {
		SetInField( N, GetField(fp));
		if (GetInField( N)) {
/*          printf("Processing an explicit semantic field.\n");*/
			GetOpenBrace(fp);
			SetCurrentField( N, NIL);
			FoundCloseBrace = FALSE;
			while ((N->frame_num < PropsInFile) && (not(FoundCloseBrace))) {
				ProcessProposition(fp, N);
				SetFrameNum( N, N->frame_num+1);
				NextChar = GetNextChar(fp);
				if (NextChar == '}') FoundCloseBrace = TRUE;
				else ungetc( NextChar, fp);
				}
/*          printf("Added the explicit semantic field:\n");*/
/*          print_explicit_field( GetCurrentField( N), N);
			strike_any_key();*/  /* pause for debugging */
			/* add the new field to the sequence of fields */
			N->fields =  conseq( GetCurrentField( N), N->fields);
			ErrorTest(
				((N->frame_num == PropsInFile) && (not(FoundCloseBrace))),
				"ERROR: Exhausted propositions without closing field.\n");
			} /* done processing an explicit semantic field */
		else { /* we're just reading a proposition */
/*          printf("Proposition is not in an explicit semantic field.\n");*/
			ProcessProposition(fp, N);
			N->frame_num++;
			}
		} /* end while not EOF . . . */
	fclose(fp); /* close the proposition file */
}


struct SEMANTIC_NETWORK *LoadSemanticNetWithPropositions( s, N)
struct SEMANTIC_NETWORK *s;     /* pointer to the semantic network */
struct NetDefStructure *N;      /* pointer to the network def structure */
{
	int the_type;
	int this_role;
	int this_mode;
	int i, j;
	int surface_position;

	SetNumNodes( s, GetNDefNumNodes( N));

	/* read the names and categories of the nodes */
	for (i=0; i < GetNDefNumNodes( N); i++) {
		strcpy( GetName( s, i), GetTheNodes( N, i));
		the_type = name_lookup( GetTheTypes( N, i), cat_names, MAX_CATS);
		if (the_type == STR_NOT_FOUND_ERROR) {
			printf("ERROR:Unrecognizable type <%s>.\n", GetTheTypes( N, i));
			exit( ERROR_VALUE);
			}
		else SetType( s, i, the_type);
		}

	/* initialize the sparse matrix for network connections */
	for (i = 0; i < MAX_NODES; i++) s->R[i] = NIL;

	/* set the number of connections in the semantic network */
	SetNumConn( s, GetNDefNumConn( N));

	/* read the connections */
	/* connection format: from_name to_name role mode surface_pos */
	for (i=0; i < GetNumConn( s); i++) {
		/* add the new connection */
		SetConn(s,
				N->connections[i].from_node,
				N->connections[i].to_node,
				i+1); /* connections start at 1 */
		SetRole( s, i+1, N->connections[i].role);
		SetMode( s, i+1, N->connections[i].mode);
		SetSurfPos( s, i+1, N->connections[i].surface_pos);
		}

	/* initialize sequence of explicit semantic fields */
	SetFields( s, N->fields);

	return( s); /* s is now initialized */
}



#define FOUND_A_FILE    0
#define MAX_FILES       30

int GetFileNamesFromDos( PropFileNames )
char PropFileNames[][MAX_STRLEN];
{

	int i;
	unsigned result;
	struct find_t FileInfoStruct;  /* defined in dos.h */

	result = _dos_findfirst( "*.prp", _A_NORMAL, &FileInfoStruct);
	if (result == FOUND_A_FILE)
		strcpy( PropFileNames[0], FileInfoStruct.name);
	else {
		printf("ERROR: No .prp files in current directory.\n");
		exit( ERROR_VALUE);
		}

	i = 1;
	while (result == FOUND_A_FILE) {
		result = _dos_findnext( &FileInfoStruct);
		if (result == FOUND_A_FILE) {
			strcpy( PropFileNames[i], FileInfoStruct.name);
			i++;
			}
		}

	return( i);  /* number of .prp files in current directory */
}

void DisplayFileNames( void)
{
	int i, NumberOfFiles, TimesInCol, row, col;
	char PropFileNames[MAX_FILES][MAX_STRLEN];

	NumberOfFiles = GetFileNamesFromDos( PropFileNames);

	/* display the file names in the file name window */
	row = 3;
	TimesInCol = 0;
	for (i = 0; i < NumberOfFiles; i++){
		col = (TimesInCol*15)+12;
		posprint(row, col, PropFileNames[i]);
		TimesInCol++;
		if (TimesInCol > 3) {
			TimesInCol = 0;
			row++;
			}
		}
}

FILE *OpenPropositionFile( fp, prop_name)
FILE *fp;
char *prop_name;
{
	int need_prp_file;

	_clearscreen( _GCLEARSCREEN);
	DisplayWindow( FileNamesWindow);
	DisplayWindow( ErrorWindow);
	DisplayFileNames();

	need_prp_file = TRUE;
	/* display initial prompt */
	_settextcursor( STANDARD_CURSOR);
	posprint(QUERY_ROW, 15,
		"Enter the name of the proposition file: [            ]");

	while (need_prp_file) {
		/* ask for the name of the proposition file */
		_settextposition(QUERY_ROW, 56);
		GetConsoleFileName( prop_name, 12);
		ClearErrorWindow();
/*      printf("prop_name = %s", prop_name);*/

		/* try to open the proposition file. */
		fp = fopen( prop_name, "r");
		if (fp == NULL) {
			PrintError("Unable to open proposition file.");
			posprint(QUERY_ROW, 15,
						"Please re-enter proposition file name: ");
			}
		else {
			posprint(QUERY_ROW+2, 15,
						"Proposition file opened successfully.\n");
			need_prp_file = FALSE; /* need satisfied */
			}
		}
	_settextcursor( NO_CURSOR);
	return( fp);
}

struct SEMANTIC_NETWORK *MakeNewSemanticNet( sem_net)
struct SEMANTIC_NETWORK *sem_net;
{
	char prop_name[MAX_STRLEN];
	FILE *fp;
	struct NetDefStructure *N;

	printf("ENTERING: MakeNewSemanticNet\n");
	/* set up the network definition structure */
	N = AllocateNetDef();
	InitializeNetDef(N);

	/* get the proposition file handle */
	fp = OpenPropositionFile( fp, prop_name);

	/* convert the proposition file to a network definition file */
	posprint(QUERY_ROW+4, 15, "Initializing the semantic network.....");
	PropFileToNetDefFile( fp, prop_name, N);

	/* initialize the semantic network data structure */
	InitializeSemanticNet( sem_net);
	sem_net = LoadSemanticNetWithPropositions( sem_net, N);
	if (null_net( sem_net )) {
		printf("ERROR:Initialization failed.\n");
		exit( ERROR_VALUE);
		}
	else posprint(QUERY_ROW+6, 15, "Initialization succeeded.");
	FreeNetDef( N);
	printf("EXITING: MakeNewSemanticNet\n");
	return( sem_net); /* return the initialized network */
}


void MakeAndUseSemanticNet( void)
{
	struct SEMANTIC_NETWORK *S;

	printf("ENTERED: MakeAndUseSemanticNet\n");
	S = AllocateSemanticNet();
	S = MakeNewSemanticNet( S);
	do_net_menu( S);
	FreeSemanticNet( S);
	printf("EXITING: MakeAndUseSemanticNet\n");
}

