/***********************************************************************/
/*                                                                     */
/*                          Inference Engine Code                      */
/*                     Copyright (C) 1993 Eric Steinhart               */
/*                                                                     */
/***********************************************************************/
#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\stdlib.h"
#include "anadefs.h"
#include "semnet.h"
#include "buffer.h"
#include "iterate.h"

#define MAX_STACK   10
#define MAX_LEVELS  15  /* maximum depth of implication complex */

struct LogicStack {
	int top;
	struct LogicNode *slots[MAX_STACK];
	};

struct LogicNode {
	int Proposition;
	char PrettyPrint[MAX_LEVELS];
	struct LogicStack *Entailments;
	struct LogicNode *Presupposition;
	};


/***********************************************************************/
/*                                                                     */
/*                     Logic Node & Stack Manipulation                 */
/*                                                                     */
/***********************************************************************/
struct LogicStack *MakeLogicStack( void )
{
	struct LogicStack *ptr;

	ptr = (struct LogicStack *)
				malloc( (size_t) sizeof( struct LogicStack));
	return( ptr);
}

void InitializeLogicStack( stack)
struct LogicStack *stack;
{
	int i;
	stack->top = 0;
	for (i=0; i < MAX_STACK; i++) {
		stack->slots[i] = NIL;
		}
}

struct LogicNode *MakeLogicNode( ThisProposition)
int ThisProposition;
{
	struct LogicNode *ptr;
	int i;

	ptr = (struct LogicNode *)
				malloc( (size_t) sizeof( struct LogicNode));
	ptr->Proposition = ThisProposition;
	for (i=0; i < MAX_LEVELS; i++) ptr->PrettyPrint[i] = NO_FILLER;
	ptr->Entailments = MakeLogicStack();
	InitializeLogicStack( ptr->Entailments);
	ptr->Presupposition = NIL;

	return( ptr);
}

void Push( pointer, stack)
struct LogicNode *pointer;
struct LogicStack *stack;
{
	if (stack->top == MAX_STACK) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR: Logic stack overflow.\n");
		exit(ERROR_VALUE);
		}

	stack->slots[stack->top] = pointer;
	stack->top++;
}

int EmptyLogicStack( stack)
struct LogicStack *stack;
{
	return( (stack->top == 0)?TRUE:FALSE);
}

void CopyLogicStack( destination, source)
struct LogicStack *destination;
struct LogicStack *source;
{
	int i;

	InitializeLogicStack(destination);

	destination->top = source->top;

	for (i=0; i < destination->top; i++) {
		destination->slots[i] = source->slots[i];
		}
}


int HasEntailments( Node)
struct LogicNode *Node;
{
	if (Node->Entailments->top != 0)
		return( TRUE);
	else return( FALSE);
}


void PrintLogicStack( S, stack )
struct SEMANTIC_NETWORK *S;
struct LogicStack *stack;
{
	int i;
	struct LogicNode *ptr;

	if (EmptyLogicStack(stack)) {
		printf("Stack is empty.\n");
		return;
		}

	for (i=0; i < stack->top; i++) {
		ptr = stack->slots[i];
		printf("%s \n", GetName(S,ptr->Proposition));
		}
}

void FreeLogicNode( Root )
struct LogicNode *Root;
{
	int i;
	struct LogicStack *Stack;

	if (Root == NIL) return;

	Stack = Root->Entailments;
	for (i=0; i < Stack->top; i++) {
		FreeLogicNode( Stack->slots[i]);
		}
	free( Stack);
	free( Root);
}


/***********************************************************************/
/*                                                                     */
/*                        Forward Chaining Functions                   */
/*                                                                     */
/***********************************************************************/
/* returns the propositions that are ENTAILED by a proposition */
cons_ptr GetEntailments( S, FrameId, Mode, Entailments)
struct SEMANTIC_NETWORK *S;
int FrameId;            /* the rule */
int Mode;               /* LITERAL or ANALOGICAL */
cons_ptr Entailments;
{
	cons_ptr rules, AnaRules, ptr, TheBranches, AnaBranches;
	int ThisRule, TheConsequent;

/*  printf("ENTERING: GetEntailments.\n");*/

	/* find all rules of which FrameId is an ANTECEDENT */
	/* each r in rules should have type RULE */
	rules = NIL;
	rules = node_trans( FrameId, S, ANTECEDENT, LITERAL, rules);
	if (Mode == ANALOGICAL) {
		AnaRules = NIL;
		AnaRules = node_trans( FrameId, S, ANTECEDENT, ANALOGICAL, AnaRules);
		rules = list_union( AnaRules, rules);
		}
/*  printf("The rules of <%s> are:\n", GetName(S,FrameId));
	print_node_names(rules, S); strike_any_key();*/

	ListIterator( ptr, rules) {
		ThisRule = car(ptr);    /* the current rule */

		TheConsequent = GetConsequent(S, ThisRule);
		switch (GetType(S,TheConsequent)) {
		  case SFRAME:
			if (not(is_member(TheConsequent, Entailments)))
				Entailments = cons(TheConsequent, Entailments);
			break;
		  case CONJUNCTION:
			TheBranches = NIL;
			TheBranches = node_trans(TheConsequent, S,
							ANDBRANCH, LITERAL, TheBranches);
			if (Mode == ANALOGICAL) {
				AnaBranches = NIL;
				AnaBranches = node_trans(TheConsequent, S,
							ANDBRANCH, ANALOGICAL, AnaBranches);
				TheBranches = list_union(AnaBranches, TheBranches);
				}
			Entailments = list_union(TheBranches, Entailments);
			break;
		  default:
			_clearscreen( _GCLEARSCREEN);
			printf("ERROR:Consequent of rule <%s> is bad grammar category.",
				GetName(S,ThisRule));
			exit(ERROR_VALUE);
			break;
		  }
		}
/*  printf("EXITING: GetEntailments.\n");*/
	return( Entailments);
}

/* forms a list of the presuppositions of Node */
cons_ptr GetPresuppositionChain( Node, Chain)
struct LogicNode *Node;
cons_ptr Chain;
{
	Chain = ncons( Node->Proposition);

	Node = Node->Presupposition;
	while (not( Node == NIL)) {
		Chain = cons(Node->Proposition, Chain);
		Node = Node->Presupposition;
		}
	return( Chain);
}


struct LogicNode *ForwardChain( S, proposition, Root, Mode)
struct SEMANTIC_NETWORK *S;
int proposition;
struct LogicNode *Root;
int Mode;                   /* LITERAL or ANALOGICAL */
{
	struct LogicStack *Leaves, *NewLeaves;
	int i;
	cons_ptr entails, presuppositions;

	Leaves = MakeLogicStack();
	InitializeLogicStack(Leaves);

	NewLeaves = MakeLogicStack();
	InitializeLogicStack(NewLeaves);

	Root = MakeLogicNode( proposition);
	Push(Root, Leaves);

	while (not(EmptyLogicStack(Leaves))) {
		InitializeLogicStack(NewLeaves);

		for (i=0; i<Leaves->top; i++) { /* treat stack as array */
			cons_ptr ptr;
			struct LogicNode *x;

			x = Leaves->slots[i];
			entails = NIL;
			entails = GetEntailments(S, x->Proposition, Mode, entails);
/*          printf("Entailments of <%s>: ", GetName(S,x->Proposition));
			print_node_names( entails, S); strike_any_key();*/

			/* find all the presuppositions of x (including x) */
			presuppositions = NIL;
			presuppositions = GetPresuppositionChain(x, presuppositions);
/*          printf("Presuppositions of <%s> are:\n",
						GetName(S,x->Proposition));
			print_node_names( presuppositions, S);
			strike_any_key();*/

			ListIterator( ptr, entails) {
				int ThisEntail;
				struct LogicNode *NewLogicNode;

				ThisEntail = car(ptr);
				if (not(is_member(ThisEntail, presuppositions))) {
					/* if is_member, we would get an infinite loop */
					NewLogicNode = MakeLogicNode( ThisEntail);
					Push(NewLogicNode, x->Entailments);
					NewLogicNode->Presupposition = x;
					Push(NewLogicNode, NewLeaves);
					}
				}
/*          printf("The NewLeaves are: \n");
			PrintLogicStack( S, NewLeaves);*/

			CopyLogicStack( Leaves, NewLeaves);
			}
		}
	return( Root);
}


/**************************************************************************/
/*                                                                        */
/*                          Printing Functions                            */
/*                                                                        */
/**************************************************************************/
#define PRINT_MARK    1

void MarkWholeTree( S, tree, mark, position)
struct SEMANTIC_NETWORK *S;
struct LogicNode *tree;
int mark;
int position;
{
	int i;
	struct LogicNode *ThisLogicNode;
	struct LogicStack *Children;

/*  printf("ENTER: MarkWholeTree.\n");*/
	Children = tree->Entailments;
	for (i=0; i < Children->top; i++) {
		/* for each child of tree, mark that child & its children */
		ThisLogicNode = Children->slots[i];
		ThisLogicNode->PrettyPrint[position] = mark;
/*      printf("Marked: %s->PrettyPrint[%d] == %d\n",
				GetName(S,ThisLogicNode->Proposition), position, mark);*/
		MarkWholeTree(S, ThisLogicNode, mark, position);
		}
/*  strike_any_key();*/
/*  printf("EXIT: MarkWholeTree.\n");*/
}

void PrettyMarkers(S, tree, mark, position)
struct SEMANTIC_NETWORK *S;
struct LogicNode *tree;
int mark;
int position;
{
	int i, TreeTop;
	struct LogicNode *ThisLogicNode;
	struct LogicStack *Children;

/*  printf("ENTER:PrettyMarkers.\n");*/

	Children = tree->Entailments;
	for (i=0; i < Children->top; i++) {
		/* for each child of tree, mark that child & its children */
		ThisLogicNode = Children->slots[i];
		ThisLogicNode->PrettyPrint[position] = mark;
/*      printf("Marked: %s->PrettyPrint[%d] == %d\n",
				GetName(S,ThisLogicNode->Proposition), position, mark);*/
		if (not(i == (Children->top - 1))) /* if not last child */
			MarkWholeTree(S, ThisLogicNode, 1, position);
		else MarkWholeTree(S, ThisLogicNode, 0, position);
		PrettyMarkers(S, ThisLogicNode, 1, position+1);
		}
/*  printf("EXIT:PrettyMarkers.\n");*/
}


void PrintComplexToBuffer( S, FB, root)
struct SEMANTIC_NETWORK *S;
struct TextBuffer *FB;
struct LogicNode *root;
{
	int i;
	struct LogicStack *stack;

	if (root == NIL) return;

	for (i=0; root->PrettyPrint[i] != NO_FILLER; i++) {
		if (root->PrettyPrint[i] == PRINT_MARK) {
			if (root->PrettyPrint[i+1] == NO_FILLER)
			   StringMove( FB, "|===> ");
			else StringMove( FB, "|    ");
			}
		else StringMove( FB, "     ");
		}

	FrameToLinePoint(S, FB, root->Proposition);

	stack = root->Entailments;
	for (i=0; i < stack->top; i++)
		/* for each child of root, print that child's tree */
		PrintComplexToBuffer(S, FB, stack->slots[i]);
}


/**************************************************************************/
/*                                                                        */
/*                          The Inference Engine                          */
/*                                                                        */
/**************************************************************************/
void InferenceEngine(S)
struct SEMANTIC_NETWORK *S;
{
	struct LogicNode *root;
	int proposition;
	int need_proposition;
	int node;
	cons_ptr Frames;
	int ThereIsAnError;
	char Temp[80];
	struct TextBuffer *FB;
	int i;

	_clearscreen( _GCLEARSCREEN);

	/*******************************************************************/
	/*            Ask the user for a frame to interpret                */
	/*******************************************************************/
	Frames = NIL;
	for (node = 0; node < GetNumNodes(S); node++) {
		if (GetType(S,node) == SFRAME)
			Frames = cons(node, Frames);
		else if (GetType(S,node) == MFRAME)
			Frames = cons(node, Frames);
		}

	need_proposition = TRUE;
	ThereIsAnError = FALSE;
	while( need_proposition) {
		_clearscreen( _GCLEARSCREEN);
		PrintListOfFrames( S, Frames, InferenceScrollWindow);
		DisplayWindow( PropNodeQueryWindow);
		DisplayWindow( ErrorWindow);
		if (ThereIsAnError) PrintError( Temp);
		/* get the proposition */
		proposition = query_user_for_proposition(S, PropNodeQueryWindow);
		_clearscreen( _GCLEARSCREEN);
		/* check if the proposition node is valid */
		if ((GetType(S,proposition) != SFRAME)
			&& (GetType(S,proposition) != MFRAME)) {
				ThereIsAnError = TRUE;
				sprintf(Temp, "Node <%s> is not a proposition.",
					GetName(S,proposition));
				}
		else need_proposition = FALSE;
		}


	/*******************************************************************/
	/*                        DO FORWARD CHAINING                      */
	/*******************************************************************/
	root = NIL;
	root = ForwardChain( S, proposition, root, LITERAL);


	/*******************************************************************/
	/*                     PRINT THE IMPLICATION COMPLEX               */
	/*******************************************************************/
	PrettyMarkers(S, root, 1, 0);  /* must apply pretty markers */
/*  PrintComplexToScreen( S, root);
	strike_any_key();*/  /* pause for debugging */

	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);

	if (root == NIL) {
		StringMove( FB, "Implication complex for <");
		StringMove( FB, GetName(S,proposition));
		StringMove( FB, "> is nil.");
		}
	else PrintComplexToBuffer( S, FB, root);
	DisplayBuffer( FB, LogicScrollWindow);
	free( FB);
}


/*************************************************************************/
/*                                                                       */
/*                      Interpret Metaphors                              */
/*                                                                       */
/*************************************************************************/
void InterpretTransferredFrame(S, map)
struct SEMANTIC_NETWORK *S;
struct map_structure *map;
{
	struct LogicNode *root;
	int TopicFrame;
	int SourceFrame;
	int need_proposition;
	int node;
	cons_ptr TransFrames, ptr;
	int ThereIsAnError;
	char Temp[80];
	struct TextBuffer *FB;
	int i;

	DisplayWindow( FieldChoiceWindow);
	DisplayExtraMsg( FieldChoiceWindow,
		"Generating Implication Complexes.....                         ");

	/* initialize the buffer for the implication complex */
	FB = AllocateTextBuffer();
	InitializeTextBuffer( FB);

	/*******************************************************************/
	/*                  Find the transferred frames                    */
	/*******************************************************************/
	TransFrames = NIL;
	TransFrames = GetTransferredTopicFrames( map, TransFrames);

	ListIterator( ptr, TransFrames) {
		/* for each transferred frame */
		TopicFrame = car(ptr);
		if ((GetType(S,TopicFrame) == SFRAME)
			|| (GetType(S,TopicFrame) == MFRAME)) {
			/**************************************************************/
			/*                   DO FORWARD CHAINING                      */
			/**************************************************************/
			root = NIL;
			root = ForwardChain( S, TopicFrame, root, ANALOGICAL);

			if (HasEntailments( root)) {
				PrettyMarkers(S, root, 1, 0); /* must apply pretty markers! */
/*              PrintComplexToScreen( S, root);
				strike_any_key();*/

				/*********************************************************/
				/*           PRINT THE IMPLICATION COMPLEX               */
				/*********************************************************/
				if (root == NIL) {
					StringMove( FB, "Implication complex for <");
					StringMove( FB, GetName(S,TopicFrame));
					StringMove( FB, "> is nil.");
					}
				else PrintComplexToBuffer( S, FB, root);
				NextLine( FB);
				} /* end if root entailments not nil */
			} /* end if SFRAME or MFRAME */
			FreeLogicNode( root);
		} /* end for each transferred frame */

	_clearscreen( _GCLEARSCREEN);
	DisplayBuffer( FB, LogicScrollWindow);
	free( FB);
}

