/***********************************************************************/
/*                                                                     */
/*                              ANALOGICAL MAP                         */
/*                Copyright (C) 1992 Eric Charles Steinhart            */
/*                            All rights reserved.                     */
/*                                                                     */
/***********************************************************************/
#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\stdlib.h"
#include "anadefs.h"
#include "semnet.h"
#include "hyponet.h"
#include "iterate.h"

#define NOT_TRANSFERRED 0
#define TRANSFERRED     1

struct pair_count {
	cons_ptr pair;      /* pointer to the analogous pair */
	int transfer_flag;  /* whether the topic node was transferred */
	};

/* the map is a table. each entry is a pair/count struct */
#define MAX_ANA_PAIRS 50
struct map_structure {
	int length;
	struct pair_count table[MAX_ANA_PAIRS];
	};


/**************************************************************************/
/*                          Map Utility Functions                         */
/**************************************************************************/
int GetMapLength( map)
struct map_structure *map;
{
	return( map->length);
}

/* retrieves the analogue of term or frame from map */
/* term or frame can be in either Target or Source  */
int get_analogue( S, map, term)
struct SEMANTIC_NETWORK *S;
struct map_structure *map;
int term;
{
	int i;

	/* search for term in Target, return analogue from Source */
	for (i=0; i<map->length; i++)
		if (car(map->table[i].pair) == term) {
/*          printf("Analogue is: %s\n",
				GetName(S,car(cdr(map->table[i].pair))));*/
			return(car(cdr(map->table[i].pair)));
			}
	/* term was not in Target */
	/* search for term in Source, return analogue from Target */
	for (i=0; i<map->length; i++)
		if (car(cdr(map->table[i].pair)) == term) {
/*          printf("Analogue is: %s\n",
				GetName(S,car(map->table[i].pair)));*/
			return(car(map->table[i].pair));
			}
	/* term was not in Target or Source, return NIL */
	return( NIL);
}

struct map_structure *TransferPairToMap( map, TF, SF)
struct map_structure *map;
int TF;
int SF;
{
	if (map->length == MAX_ANA_PAIRS) {
		_clearscreen( _GCLEARSCREEN);
		printf("ERROR: Too many analogous pairs in map.\n");
		printf("       Cannot transfer new pair to map.\n");
		exit( ERROR_VALUE);
		}
	map->length++;
	map->table[map->length].pair = cons(TF, ncons(SF));
	map->table[map->length].transfer_flag = TRANSFERRED;
	return( map);
}

cons_ptr GetTransferredTopicFrames( map, TransferredTopicFrames)
struct map_structure *map;
cons_ptr TransferredTopicFrames;
{
	int i;

	for (i=0; i <= map->length; i++)
		if (map->table[i].transfer_flag == TRANSFERRED)
			TransferredTopicFrames
				= cons( car(map->table[i].pair),
							TransferredTopicFrames);
	return( TransferredTopicFrames);
}

/* returns TRUE if every x in list has an analogue */
int AllHaveAnalogues( S, map, list)
struct SEMANTIC_NETWORK *S;
struct map_structure *map;
cons_ptr list;
{
	cons_ptr ptr;
	int element;

	ListIterator( ptr, list) {
		element = car(ptr);
		if (get_analogue(S, map, element) == NIL) return( FALSE);
		}
	return( TRUE);
}



/**************************************************************************/
/*                         Map Building Functions                         */
/**************************************************************************/
struct map_structure *AllocateMap( void)
{
	struct map_structure *MapPtr;

	MapPtr = (struct map_structure *)
				malloc( (size_t) sizeof( struct map_structure));
	if (MapPtr == NULL) {
		printf("ERROR: Can't allocate analogical map.\n");
		printf("ERROR: Out of memory.\n");
		exit( ERROR_VALUE);
		}
	return( MapPtr);
}

void InitializeMap( map)
struct map_structure *map;
{
	int i;

	/* initialize the map */
	for (i=0; i < MAX_ANA_PAIRS; i++) {
		map->table[i].pair = NIL;
		map->table[i].transfer_flag = NOT_TRANSFERRED;
		}
	map->length = 0;  /* initial length of the map is zero */
	return( map);
}

/* builds a map from a sequence of match hypotheses */
struct map_structure *MakeMap( map, NodePairs)
struct map_structure *map;
sequence_ptr NodePairs;
{
	sequence_ptr ptr;
	cons_ptr winning_pair; /* a winning match hypothesis */
	int i;

	i = 0;  /* index into the map table */
	/* for every winning match hypothesis winning_pair do */
	/*      add winning_pair to the map; */
	SequenceIterator( ptr, NodePairs) {
		winning_pair = carseq(ptr);
		map->table[i].pair = winning_pair;
		map->table[i].transfer_flag = NOT_TRANSFERRED;
		i++;
		if (i > MAX_ANA_PAIRS) {
			printf("ERROR: Too many analogous pairs.\n");
			exit( ERROR_VALUE);
			}
		}
	map->length = i;
	return( map);
}

struct map_structure *make_map_from_net( S, map, target, source)
struct SEMANTIC_NETWORK *S;
struct map_structure *map;
cons_ptr target;
cons_ptr source;
{
	sequence_ptr TheWinners;

	InitializeMap( map);

	/* run a hypothesis network to find winning match hypotheses */
	TheWinners = NIL;
	TheWinners = RunHypothesisNet( S, target, source, TheWinners);

	/* make the map */
	map = MakeMap( map, TheWinners);

	return( map);
}


/* This function is called from the main menu. */
void mapping_with_network( S )
struct SEMANTIC_NETWORK *S;     /* the semantic network */
{
	int i, target_cue, source_cue;
	cons_ptr target_field, source_field;
	int need_target, need_source, ThereIsAnError;
	char ErrorMsg[80];
	sequence_ptr TheWinners;

	struct map_structure *map;
	map = AllocateMap();
	InitializeMap( map);

	_clearscreen( _GCLEARSCREEN);
	need_target = TRUE;
	ThereIsAnError = FALSE;
	while( need_target) {
		_clearscreen( _GCLEARSCREEN);
		DisplayTermsInNetwork( S, TermsWindow);
		DisplayWindow( TopicNodeQueryWindow);
		DisplayWindow( ErrorWindow);
		if (ThereIsAnError) PrintError( ErrorMsg);
		/* get the target cue */
		target_cue = query_user_for_node(S, TopicNodeQueryWindow);
		_clearscreen( _GCLEARSCREEN);
		/* generate target field from the cue */
		target_field = NIL;
		target_field = find_one_field( S, target_cue, target_field);
		if (null( target_field)) {
			ThereIsAnError = TRUE;
			sprintf(ErrorMsg, "Target field for <%s> was NIL.",
					GetName(S,target_cue));
			}
		else need_target = FALSE;
		}

	need_source = TRUE;
	ThereIsAnError = FALSE;
	while( need_source) {
		_clearscreen( _GCLEARSCREEN);
		DisplayTermsInNetwork( S, TermsWindow);
		DisplayWindow( SourceNodeQueryWindow);
		DisplayWindow( ErrorWindow);
		if (ThereIsAnError) PrintError( ErrorMsg);
		/* get the source cue */
		source_cue = query_user_for_node(S, SourceNodeQueryWindow);
		_clearscreen( _GCLEARSCREEN);
		/* generate source field from the cue */
		source_field = NIL;
		source_field = find_one_field( S, source_cue, source_field);
		if (null( source_field)) {
			ThereIsAnError = TRUE;
			sprintf(ErrorMsg, "Source field for <%s> was NIL.",
					GetName(S,source_cue));
			}
		else need_source = FALSE;
		}

	/* run a hypothesis network to find winning match hypotheses */
	TheWinners = NIL;
	TheWinners = RunHypothesisNet(
					S, target_field, source_field, TheWinners);

	/* make the map */
	map = MakeMap( map, TheWinners);

	free( map);
}


