/***************************************************************************/
/*                                                                         */
/*                        SEMANTIC NETWORK FUNCTIONS                       */
/*                                                                         */
/***************************************************************************/
#include "stdio.h"
#include "anadefs.h"
#include "C:\qc25\include\stdlib.h"

/***************************************************************************/
/*                                 DEFINITIONS                             */
/***************************************************************************/
/* Define the maximum number of nodes. */
#define MAX_NODES  400

/* Define the maximum number of connections. */
#define MAX_CONNECTIONS  1500

/* The enumerated types VERB . . . WHOLE and MAX_FUNC_ROLES */
/* and MAX_ROLES are correlated with the role names file <SemRoles.txt>. */
/* <SemRoles.txt> is browsable but not modifiable by the user. */

/* THEMATIC_ROLES is an enumerated type. */
#define VERB            0
#define AGENT           1
#define PATIENT         2
#define RECIPIENT       3
#define SOURCE          4
#define INSTRUMENT      5
#define GOAL            6
#define PATH            7
#define ABOUT           8   /* FIRST_PREPOSITION */
#define ABOVE           9
#define AROUND         10
#define AT             11
#define AWAY           12
#define AWAY_FROM      13
#define BELOW          14
#define BENEATH        15
#define BESIDE         16
#define BETWEEN        17
#define BY             18
#define DOWN           19
#define FOR            20
#define FROM           21
#define IN             22
#define INSIDE         23
#define INSIDE_OF      24
#define NEAR           25
#define OF             26
#define OFF            27
#define ON             28
#define ONTO           29
#define OUT            30
#define OUTSIDE_OF     31
#define OVER           32
#define THROUGH        33
#define THROUGHOUT     34
#define TO             35
#define TOWARD         36
#define UNDER          37
#define UNDERNEATH     38
#define UP             39
#define WITH           40
#define WITHIN         41
#define WITHOUT        42
#define M_VERB         43
#define M_AGENT        44
#define M_PATIENT      45
#define M_RECIPIENT    46
#define M_SOURCE       47
#define M_INSTRUMENT   48
#define M_GOAL         49
#define M_PATH         50   /* MAX_FUNC_ROLES */
#define PART           51
#define WHOLE          52
#define GEN1           53
#define GEN2           54
#define SUBJECT        55
#define PREDICATE      56
#define ANTECEDENT     57
#define CONSEQUENT     58
#define ANDBRANCH      59
#define SUPERTYPE      60
#define SUBTYPE        61
#define ATTRIBUTE      62
#define VALUE          63
#define PROPERTY       64
#define OBJECT         65
#define SUCCESSOR      66
#define PREDECESSOR    67
#define CONTRAST       68   /* FIRST_PARA_ROLE */
#define POSITIVE       69   /* FIRST_PARA_ARG_ROLE */
#define NEGATIVE       70   /* LAST_PARA_ROLE */

/******************************************************************/
/*                                                                */
/* WARNING! WARNING! WARNING! WARNING! WARNING! WARNING! WARNING! */
/*                                                                */
/* The way roles are now structured (t+MOD_ROLE) does not yield   */
/* %t for prepositions!   (ABOVE+MOD_ROLE) == PART                */
/*                                                                */
/* WARNING! WARNING! WARNING! WARNING! WARNING! WARNING! WARNING! */
/******************************************************************/

/* Roles from FIRST_MOD_ROLE to MAX_FUNC_ROLES are modifier roles */
/* If t is a non-mod role, (t+MOD_ROLE) is %t */
/* where %t is the modifier role for t */
/* I.e. %AGENT is the modifier role for AGENT */
#define MOD_ROLES               43
#define FIRST_PREPOSITION        8
/* Roles above MAX_FUNC_ROLES are non-functional roles */
#define MAX_FUNC_ROLES          50
/* Roles from FIRST_PARA_ROLE to LAST_PARA_ROLE are paradigmatic roles */
#define FIRST_PARA_ROLE         68
#define FIRST_PARA_ARG_ROLE     69
#define LAST_PARA_ROLE          70

/* Define maximum number of thematic roles */
#define MAX_ROLES               71

/* SEMANTIC_MODES is an enumerated type */
#define LITERAL         0
#define ANALOGICAL      1
#define MAX_MODES       2


/* GRAMMATICAL_CATEGORIES is an enumerated type. */
#define NOUN            0
#define VERBTYPE        1
#define ADJECTIVE       2
#define ADVERB          3
#define CONTRASTTYPE    4
#define SFRAME          5
#define MFRAME          6
#define PFRAME          7
#define RULE            8
#define CONJUNCTION     9
#define PROPERTYTYPE    10
#define DESCRIPTION     11
#define MAX_CATS        12

#define NET_NIL_CODE -1

extern char role_names[MAX_ROLES][MAX_STRLEN] = {
	#include "SemRoles.txt"
	};

extern char mode_names[MAX_MODES][MAX_STRLEN]
	= {"LITERAL", "ANALOGICAL"};

extern char cat_names[MAX_CATS][MAX_STRLEN] = {
	"NOUN", "VERBTYPE", "ADJECTIVE", "ADVERBTYPE", "CONTRASTTYPE",
	"SFRAME", "MFRAME", "PFRAME", "RULE", "CONJUNCTION",
	"PROPERTYTYPE", "DESCRIPTION"};

/* define a cell for representing a connection */
struct connect_node {
	int col;
	int val;
	struct connect_node *next;
	};
typedef struct connect_node *connect_ptr;


struct SEMANTIC_NETWORK {
	int             num_nodes;
	char            node_names[MAX_NODES][MAX_STRLEN];
	int             num_connections;
	connect_ptr     R[MAX_NODES];  /* a sparse matrix */
	int             role[MAX_CONNECTIONS];
	int             mode[MAX_CONNECTIONS];
	int             surface_pos[MAX_CONNECTIONS];
	int             node_type[MAX_NODES];
	sequence_ptr    fields;
	};

#define GetNumNodes( S)         (S->num_nodes)
#define SetNumNodes( S, n)      (S->num_nodes = n)
#define GetNodeNames(S)         (S->node_names)
#define GetName( S, n)          (S->node_names[n])
#define GetNumConn( S)          (S->num_connections)
#define SetNumConn( S, n)       (S->num_connections = n);
#define GetRole( S, n)          (S->role[n])
#define SetRole( S, n, r)       (S->role[n] = r)
#define GetMode( S, n)          (S->mode[n])
#define SetMode( S, n, m)       (S->mode[n] = m)
#define GetSurfPos( S, n)       (S->surface_pos[n])
#define SetSurfPos( S, n, p)    (S->surface_pos[n] = p)
#define GetType( S, n)          (S->node_type[n])
#define SetType( S, n, t)       (S->node_type[n] = t)
#define GetFields( S)           (S->fields)
#define SetFields( S, n)        (S->fields = n)

/*************************************************************************/
/*                 Allocation and Initialization Functions               */
/*************************************************************************/
struct SEMANTIC_NETWORK *AllocateSemanticNet( void)
{
	struct SEMANTIC_NETWORK *SemNetPtr;

	SemNetPtr = (struct SEMANTIC_NETWORK *)
					malloc((size_t) sizeof( struct SEMANTIC_NETWORK));
	if (SemNetPtr == NULL) {
		printf("ERROR: Can't allocate semantic network structure.\n");
		printf("ERROR: Out of memory.\n");
		exit( ERROR_VALUE);
		}
	return( SemNetPtr);
}

void FreeConnectionList( c)
connect_ptr c;
{
	connect_ptr x;

	if (c == NIL) return;
	x = c->next;
	free( c);
	FreeConnectionList( x);
}

void FreeSemanticNet( S)
struct SEMANTIC_NETWORK *S;
{
	int i;
	for (i = 0; i < MAX_NODES; i++) FreeConnectionList( S->R[i]);
/*  FreeSequence( S->fields);*/
	free( S);
}

struct SEMANTIC_NETWORK *InitializeSemanticNet( S)
struct SEMANTIC_NETWORK *S;
{
	int i, j, k;
	connect_ptr ptr;

	S->num_nodes = NET_NIL_CODE;

	for (i = 0; i < MAX_NODES; i++) S->node_names[i][0] = (char) 0;

	S->num_connections = NET_NIL_CODE;

	for (i = 0; i < MAX_NODES; i++) S->R[i] = NIL;

	for (i = 0; i < MAX_CONNECTIONS; i++) {
		S->role[i] = 0;
		S->mode[i] = 0;
		S->surface_pos[i] = 0;
		}

	for (i = 0; i < MAX_NODES; i++) S->node_type[i] = 0;

	S->fields = NIL;

/*  printf("EXIT:InitializeSemanticNet\n");*/
	return( S);
}

/* This function tests whether a semantic network is nil. */
int null_net( s)
struct SEMANTIC_NETWORK *s;
{
	if ((s->num_nodes == NET_NIL_CODE) &
		(s->num_connections == NET_NIL_CODE))
		return( TRUE);
	else return( FALSE);
}


/***********************************************************************/
/*                   Connection Matrix Get & Set Functions             */
/*                 The connection matrix is a sparse matrix.           */
/***********************************************************************/
int GetConn(S, row, column)
struct SEMANTIC_NETWORK *S;
int row;
int column;
{
	connect_ptr ptr;

	for (ptr = S->R[row]; ptr != NIL; ptr=ptr->next) {
		if (ptr->col == column) return( ptr->val);
		}
	return( NIL);
}

connect_ptr MakeConnectNode( column, value, next_node, result)
int column;
int value;
connect_ptr next_node;
connect_ptr result;
{
	result = (connect_ptr)
				   malloc((size_t) sizeof( struct connect_node));
	if (result == NULL) {
		printf("ERROR: Can't allocate connection node.\n");
		printf("ERROR: Out of memory.\n");
		exit( ERROR_VALUE);
		}
	result->col = column;
	result->val = value;
	result->next = next_node;
	return( result);
}


void SetConn(S, row, column, value)
struct SEMANTIC_NETWORK *S;
int row;
int column;
int value;
{

	int i;
	connect_ptr ptr;

	if (row > MAX_NODES) {
		printf("ERROR:Connection matrix row index out of bounds.\n");
		exit(ERROR_VALUE);
		}

	if (column > MAX_NODES) {
		printf("ERROR:Connection matrix column index out of bounds.\n");
		exit(ERROR_VALUE);
		}

	if (value == NIL) return;

	if (S->R[row] == NIL) { /* row with no entries */
		S->R[row] = MakeConnectNode(column, value, NIL, S->R[row]);
		return;
		}

	for (ptr=S->R[row]; ptr != NIL; ptr=ptr->next) {
		if (ptr->col == column) {
			ptr->val = value;
			return;
			}
		}

	for (ptr=S->R[row]; ptr->next != NIL; ptr=ptr->next);
	ptr->next = MakeConnectNode(column, value, NIL, ptr->next);
	return;
}

/* checks for the existence of a t connection between x and y */
/* returns TRUE if there is no t connection between x and y */
/* returns FALSE otherwise */
int no_connection( S, x, y, t)
struct SEMANTIC_NETWORK *S;
int x;
int y;
int t;          /* the thematic role */
{
	if (GetConn(S, x, y) == NIL)
		return( TRUE);
	else if (S->role[GetConn(S, x, y)] != t)
			return( TRUE);
		else return( FALSE);
}


struct SEMANTIC_NETWORK *AddNode(S, NodeType, NodeName)
struct SEMANTIC_NETWORK *S;
int NodeType;     /* grammatical category of the node */
char *NodeName;   /* name of the node */
{
	if (S->num_nodes >= MAX_NODES) {
		printf("ERROR: Exceeded maximum number of nodes.\n");
		exit(ERROR_VALUE);
		}
	strcpy(S->node_names[S->num_nodes], NodeName);
	S->node_type[S->num_nodes] = NodeType;
/*  printf("Added a new node: <%s>.\n", S->node_names[S->num_nodes]);*/
	S->num_nodes++;
	return(S); /* return the modified network */
}

/* adds a connection of role <t> and order <order> from x to y */
/* adds ONLY THIS CONNECTION; this is NOT SYMMETRIC */
struct SEMANTIC_NETWORK *AddOneConnection( S, x, y, t, order)
struct SEMANTIC_NETWORK *S;
int x;
int y;
int t;                  /* the thematic role */
int order;              /* LITERAL or ANALOGICAL */
{
	/* add a new connection from x to y */
	/* the connection is an <order> <t> connection */
	S->num_connections++;
	if (S->num_connections > MAX_CONNECTIONS) {
		printf("ERROR:Exceeded maximum number of connections.\n");
		exit( ERROR_VALUE);
		}
	/* add connection (x, y) */
	SetConn(S, x, y, S->num_connections);
	/* set the role of the connection to t */
	S->role[S->num_connections] = t;
	/* set the mode of the connection to <order> */
	S->mode[S->num_connections] = order;
	/* set the surface position to 0 */
	S->surface_pos[S->num_connections] = 0;

	/* notify of the new connection */
/*  printf("Added a <%s> <%s> connection from <%s> to <%s>.\n",
		role_names[t], (order == LITERAL)?"LITERAL":"ANALOGICAL",
		S->node_names[y], S->node_names[x]);*/

	return( S); /* return the modified semantic net */
}


/* adds a connection of role <t> and order <order> between x and y */
/* adds both the (x,y) connection and the (y,x) connection */
struct SEMANTIC_NETWORK *AddConnections( S, x, y, t, order, position)
struct SEMANTIC_NETWORK *S;
int x;
int y;
int t;                  /* the thematic role */
int order;              /* LITERAL or ANALOGICAL */
int position;           /* the surface position */
{
	/* add a new connection from x to y */
	/* the connection is an <order> <t> connection */
	S->num_connections++;
	if (S->num_connections > MAX_CONNECTIONS) {
		printf("ERROR:Exceeded maximum number of connections.\n");
		exit( ERROR_VALUE);
		}
	/* add connection (x, y) */
	SetConn(S, x, y, S->num_connections);
	/* set the role of the connection to t */
	S->role[S->num_connections] = t;
	/* set the mode of the connection to <order> */
	S->mode[S->num_connections] = order;
	/* set the surface position of the connection to <position> */
	S->surface_pos[S->num_connections] = position;

	/* add a new connection from y to x */
	/* the connection is an <order> <t> connection */
	S->num_connections++;
	if (S->num_connections > MAX_CONNECTIONS) {
		printf("ERROR:Exceeded maximum number of connections.\n");
		exit( ERROR_VALUE);
		}
	/* add connection (y,x) */
	SetConn(S, y, x, S->num_connections);
	/* set the role of the connection to t */
	S->role[S->num_connections] = t;
	/* set the mode of the connection to <order> */
	S->mode[S->num_connections] = order;
	/* set the surface position of the connection to <position> */
	S->surface_pos[S->num_connections] = position;

	/* notify of the new connections */
/*  printf("Added a <%s> <%s> connection between <%s> and <%s>.\n",
		role_names[t], (order == LITERAL)?"LITERAL":"ANALOGICAL",
		S->node_names[y], S->node_names[x]);*/
	return( S); /* return the modified semantic net */
}

/* prints the members of a list as named nodes */
int print_node_names( list, S)
cons_ptr list;
struct SEMANTIC_NETWORK *S;
{
	cons_ptr t;
	t = list;
	if ( null( t))
		printf("NIL\n");
	else while ( not( null( t))) {
		printf( "%s ", S->node_names[ car( t)]);
		t = cdr( t);
		}
	printf("\n");
}


/* prints the members of a list as named nodes */
int print_node_names_NOCR( list, S)
cons_ptr list;
struct SEMANTIC_NETWORK *S;
{
	cons_ptr t;
	t = list;
	if ( null( t))
		printf("NIL\n");
	else while ( not( null( t))) {
		printf( "%s ", S->node_names[ car( t)]);
		t = cdr( t);
		}
}

/* prints the members of a sequence as lists of named nodes */
void print_sequence_names( sequence, sem_net)
sequence_ptr sequence;
struct SEMANTIC_NETWORK *sem_net;
{
	sequence_ptr s;
	s = sequence;
	if ( nullseq( s))
		printf("NIL");
	else while ( not( nullseq( s))) {
		printf("\t");
		print_node_names( carseq( s), sem_net);
		printf("\n");
		s = cdrseq( s);
		}
	printf("\n");
}

