/***********************************************************************/
/*                                                                     */
/*                           TRANSITION FUNCTIONS                      */
/*                Copyright (C) 1992 Eric Charles Steinhart            */
/*                            All rights reserved.                     */
/*                                                                     */
/***********************************************************************/

#include "stdio.h"
#include "C:\qc25\include\graph.h"
#include "C:\qc25\include\dos.h"
#include "C:\qc25\include\stdlib.h"
#include "anadefs.h"
#include "semnet.h"
#include "iterate.h"

/* returns TRUE if term is an adjective */
/* returns FALSE otherwise */
is_adjective( S, term)
struct SEMANTIC_NETWORK *S;
int term;
{
	if (GetType( S,term) == ADJECTIVE)
		return(TRUE);
	else return(FALSE);
}

int is_a_term( S, y )
struct SEMANTIC_NETWORK *S;
int y;
{
	switch (GetType( S,y)) {
		case SFRAME: return( FALSE);
		case MFRAME: return( FALSE);
		case PFRAME: return( FALSE);
		default: return( TRUE);
		}
}

int is_plural( word )
char *word;
{
	if ((word[strlen(word)-1] == 's') && islower(word[0]))
		return( TRUE);
	else return( FALSE);
}

is_vowel( word)
char *word;
{
	switch (word[0]) {
		case 'a': return( TRUE);
		case 'e': return( TRUE);
		case 'i': return( TRUE);
		case 'o': return( TRUE);
		case 'u': return( TRUE);
		default: return( FALSE);
		}
}

int is_attribute(S, Noun)
struct SEMANTIC_NETWORK *S;
int Noun;
{
	cons_ptr attributes;

	attributes = NIL;
	attributes = node_trans( Noun, S, ATTRIBUTE, LITERAL, attributes);

	return( not(null(attributes)));
}


/***************************************************************************/
/***************************************************************************/
/* Propagate activation from origin to all nodes to which origin */
/* is connected by functional roles */
/*                        IMPORTANT NOTE !                      */
/*                        IMPORTANT NOTE !                      */
/* gen_node_trans() does NOT spread activation along structural */
/* roles; it only spreads activation along functional roles     */
cons_ptr gen_node_trans( origin, S, sm, to)
int origin;                    /* origin is a node in the network */
struct SEMANTIC_NETWORK *S;    /* S is the semantic network */
int sm;                        /* sm is a semantic mode */
cons_ptr to;                   /* to is a list of network nodes */
{
	int connection; /* the retrieved connection */
	int i;          /* node to which origin is connected */
	for (i=0; i < GetNumNodes( S); i++) {
		connection = GetConn(S, origin, i);
		if ( not( connection == NIL)) {
			/* then there is a connection from origin to i */
/*                      printf("Connection to: %d\n", i); */
			if ( GetRole( S, connection) <= MAX_FUNC_ROLES)
			    /* then the role is functional */
				if ( GetMode( S, connection) == sm)
				to = list_union( ncons( i), to);
			}
		} /* end for i */
	return( to);
}


/* Propagate activation from origin to all nodes to which origin */
/* is connected by connections with role tr and mode sm */
cons_ptr node_trans( origin, S, tr, sm, to)
int origin;                    /* origin is a node in the network */
struct SEMANTIC_NETWORK *S;    /* S is the semantic network */
int tr;                        /* tr is a thematic role */
int sm;                        /* sm is a semantic mode */
cons_ptr to;                   /* to is a list of network nodes */
{
	int connection; /* the retrieved connection */
	int i;          /* node to which origin is connected */
	for (i=0; i < GetNumNodes( S); i++) {
		connection = GetConn(S, origin, i);
		if ( not( connection == NIL)) {
			/* then there is a connection from origin to i */
			/* printf("Connection to: %d\n", i); */
			if ( GetMode( S, connection) == sm)
			   if (GetRole( S, connection) == tr)
				to = list_union( ncons( i), to);
			}
		} /* end for i */
	return( to);
}

/* Propagate activation from origin to all nodes to which origin */
/* is connected by connections with nominal roles and mode sm */
cons_ptr nom_node_trans( origin, S, sm, to)
int origin;                    /* origin is a node in the network */
struct SEMANTIC_NETWORK *S;    /* S is the semantic network */
int sm;                        /* sm is a semantic mode */
cons_ptr to;                   /* to is a list of network nodes */
{
	int connection; /* the retrieved connection */
	int i;          /* node to which origin is connected */
	for (i=0; i < GetNumNodes( S); i++) {
		connection = GetConn(S, origin, i);
		if ( not( connection == NIL)) {
			/* then there is a connection from origin to i */
			/* printf("Connection to: %d\n", i); */
			if ( GetMode( S, connection) == sm)
			   if ( GetRole( S, connection) != VERB)
				to = list_union( ncons( i), to);
			}
		} /* end for i */
	return( to);
}

/* Propagate activation from origin to all nodes to which origin */
/* is connected by connections with paradigmatic roles and mode sm */
cons_ptr para_node_trans( origin, S, sm, to)
int origin;                    /* origin is a node in the network */
struct SEMANTIC_NETWORK *S;    /* S is the semantic network */
int sm;                        /* sm is a semantic mode */
cons_ptr to;                   /* to is a list of network nodes */
{
	int connection; /* the retrieved connection */
	int i;          /* node to which origin is connected */
	for (i=0; i < GetNumNodes( S); i++) {
		connection = GetConn(S, origin, i);
		if ( not( connection == NIL)) {
			/* then there is a connection from origin to i */
			/* printf("Connection to: %d\n", i); */
			if ( GetMode( S, connection) == sm)
			   if ((FIRST_PARA_ROLE <= GetRole( S, connection)) &&
				(GetRole( S,connection) <= LAST_PARA_ROLE))
				to = list_union( ncons( i), to);
			}
		} /* end for i */
	return( to);
}

/* propagate activation from state to all nodes to which nodes in */
/* state are connected regardless of role */
cons_ptr gen_state_trans( state, S, sm, to)
cons_ptr state;                 /* state is a list of nodes */
struct SEMANTIC_NETWORK *S;     /* S is a semantic network */
int sm;                         /* sm is the semantic mode */
cons_ptr to;                    /* to is the activated nodes */
{
	cons_ptr s, temp;
	s = state;
	while( not( null( s))) {
		/* spread activation from s */
		temp = NIL;
		temp = gen_node_trans( car( s), S, sm, temp);
		to = list_union( temp, to);
		s = cdr( s);
		}
	return( to);
}

/* propagate activation from state to all nodes to which nodes in */
/* state are connected by connections with role tr and mode sm */
cons_ptr state_trans( state, S, tr, sm, to)
cons_ptr state;                 /* state is a list of nodes */
struct SEMANTIC_NETWORK *S;     /* S is a semantic network */
int tr;                         /* tr is the thematic role */
int sm;                         /* sm is the semantic mode */
cons_ptr to;                    /* to is the activated nodes */
{
	cons_ptr s, temp;
	s = state;
	while( not( null( s))) {
		/* spread activation from s */
		temp = NIL;
		temp = node_trans( car( s), S, tr, sm, temp);
		to = list_union( temp, to);
		s = cdr( s);
		}
	return( to);
}

/* propagate activation from state to all nodes to which nodes in */
/* state are connected by nominal connections with mode sm */
cons_ptr nom_state_trans( state, S, sm, to)
cons_ptr state;                 /* state is a list of nodes */
struct SEMANTIC_NETWORK *S;     /* S is a semantic network */
int sm;                         /* sm is the semantic mode */
cons_ptr to;                    /* to is the activated nodes */
{
	cons_ptr s, temp;
	s = state;
	while( not( null( s))) {
		/* spread activation from s */
		temp = NIL;
		temp = nom_node_trans( car( s), S, sm, temp);
		to = list_union( temp, to);
		s = cdr( s);
		}
	return( to);
}

/**************************************************************************/
/*                                                                        */
/*                          Mereological Functions                        */
/*                                                                        */
/**************************************************************************/
/* WholeOf(x) returns y such that y contains x */
/* or returns NO_WHOLE if there is no such y */
int WholeOf(S,x)
struct SEMANTIC_NETWORK *S;
int x;
{
	cons_ptr the_wholes;

	/* find y such that x is a PART of y */
	the_wholes = NIL;
	the_wholes = node_trans( x, S, PART, LITERAL, the_wholes);
/*      printf("IN WholeOf: WholesOf %s are:", GetName( S,x));
	print_node_names( the_wholes, S);*/

	/* WholeOf can't return NIL since there is a node */
	/* whose index in the semantic network is 0 */
	/* hence a special code NO_WHOLE is necessary. */
	if (null( the_wholes)) return( NO_WHOLE);

	if (length( the_wholes) > 1)
		printf("WARNING: Many containing wholes for %s.\n",
			GetName( S,x));

	return( car( the_wholes));
}

int MaximalWholeOf( S, x)
struct SEMANTIC_NETWORK *S;
int x;                  /* a node in the semantic network */
{

	while(WholeOf(S,x) != NO_WHOLE) {
		x = WholeOf(S,x); /* up mereological hierarchy */
		}
	return( (x==NO_WHOLE)?NIL:x);
}

cons_ptr MereologicalChain( S, x, result)
struct SEMANTIC_NETWORK *S;
int x;                  /* a node in the semantic network */
cons_ptr result;        /* the mereological chain generated by x */
{

/*      printf("ENTER: MereologicalChain.\n");*/
	while(WholeOf(S,x) != NO_WHOLE) {
/*              printf("Whole of %s is %s\n",
			GetName( S,x), GetName( S,WholeOf(S,x)));*/
		result = list_append( result, WholeOf(S,x));
		x = WholeOf(S,x); /* up mereological hierarchy */
/*              if (x == NO_WHOLE) printf("Top reached.\n");
		else {  printf("x is %d\n", x);
			printf("x is %s\n", GetName( S,x)); }*/
		}

/*      printf("EXIT: MereologicalChain.\n");*/
	return( result);
}

/* finds all the parts of x.  NOTICE that x is not a part of x */
/* so x is not in the list returned by AllPartsOf(x) */
cons_ptr AllPartsOf(S, x, result)
struct SEMANTIC_NETWORK *S;
int x;
cons_ptr result;
{
	cons_ptr parts, ptr;
	int y;

	/* spread activation to all y such that x is a WHOLE of y */
	parts = NIL;
	parts = node_trans( x, S, WHOLE, LITERAL, parts);
/*  printf("Parts of <%s> are: ", GetName( S,x));
	print_node_names( parts, S);*/

	/* if x has parts, process those parts */
	ptr = parts;
	while (not( null( ptr))) {
		y = car(ptr);
		result = cons( y, result);
		result = AllPartsOf(S, y, result);
		ptr = cdr(ptr);
		}
	return( result);
}


/**************************************************************************/
/*                                                                        */
/*                Utility Functions for Surface Positions                 */
/*                                                                        */
/**************************************************************************/
/* returns TRUE if Connection has a noun modifier role */
/* returns FALSE otherwise */
int has_noun_modifier_role( S, Connection)
struct SEMANTIC_NETWORK *S;
int Connection;
{
	if ((GetRole( S,Connection) > MOD_ROLES) &&
		(GetRole( S,Connection) <= MAX_FUNC_ROLES))
		return( TRUE);
	else return( FALSE);
}

/* returns TRUE if Connection has a noun modifier role */
/* returns FALSE otherwise */
int has_modifier_role( S, Connection)
struct SEMANTIC_NETWORK *S;
int Connection;
{
	if ((GetRole( S,Connection) >= MOD_ROLES) &&
		(GetRole( S,Connection) <= MAX_FUNC_ROLES))
		return( TRUE);
	else return( FALSE);
}

/* returns the ADJECTIVE at a position if there is one */
/* returns NO_FILLER otherwise */
int GetAdjectiveAtPosition(S, FrameId, Position)
struct SEMANTIC_NETWORK *S;
int FrameId;
int Position;
{
	int Node, Connection;

	/* search for a connection with surface_pos == position */
	/* and with role == a modifier role */
	/* return the modifier at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		Connection = GetConn(S, FrameId, Node);
		if (Connection != NIL) {
			if ((GetSurfPos( S,Connection) == Position) &&
					(has_noun_modifier_role(S, Connection))) {
						return( Node);
				}
			}
		}
	return( NO_FILLER);
}

/* returns the ADJECTIVE or ADVERB at a position if there is one */
/* returns NO_FILLER otherwise */
int GetModifierAtPosition(S, FrameId, Position)
struct SEMANTIC_NETWORK *S;
int FrameId;
int Position;
{
	int Node, Connection;

	/* search for a connection with surface_pos == position */
	/* and with role == a modifier role */
	/* return the modifier at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		Connection = GetConn(S, FrameId, Node);
		if (Connection != NIL) {
			if ((GetSurfPos( S,Connection) == Position) &&
					(has_modifier_role(S, Connection))) {
						return( Node);
				}
			}
		}
	return( NO_FILLER);
}

/* returns the noun at a position if there is one */
/* returns NO_FILLER otherwise */
int GetNounAtPosition(S, FrameId, Position)
struct SEMANTIC_NETWORK *S;
int FrameId;
int Position;
{
	int Node, Connection;

	/* search for a connection with surface_pos == position */
	/* and which is directed to a NOUN */
	/* return the noun at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		Connection = GetConn(S, FrameId, Node);
		if (Connection != NIL) {
			if ((GetSurfPos( S,Connection) == Position) &&
					(GetType( S,Node) == NOUN)) {
						return( Node);
				}
			}
		}
	return( NO_FILLER);
}


/* returns the verb of a syntagmatic frame */
int GetFrameVerb( S, FrameId)
struct SEMANTIC_NETWORK *S;
int FrameId;
{
	int Node, Connection;

	/* search for a connection with VERB as its role */
	/* return the verb at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		Connection = GetConn(S, FrameId, Node);
		if (Connection != NIL) {
			if (GetRole( S,Connection) == VERB) return( Node);
			}
		}
	return( NO_FILLER);
}

/* returns NOUN, VERB, PROPERTY, SFRAME at a position if there is one */
/* returns NO_FILLER otherwise */
int GetHeadAtPosition(S, FrameId, Position)
struct SEMANTIC_NETWORK *S;
int FrameId;
int Position;
{
	int Node, Connection;

	/* search for a connection with surface_pos == position */
	/* and which is directed to a NOUN, VERB, PROPERTY, or DESCRIPTION */
	/* return the noun at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		Connection = GetConn(S, FrameId, Node);
		if (Connection != NIL) {
			if (GetSurfPos( S,Connection) == Position) {
				switch (GetType( S, Node)) {
					case NOUN:
					case PROPERTYTYPE:
					case SFRAME:
					case VERBTYPE:
					case DESCRIPTION:
						 return( Node);
						 break;
					}
				}
			}
		}
	return( NO_FILLER);
}


int GetRoleOfPosition(S, FrameId, Position)
struct SEMANTIC_NETWORK *S;
int FrameId;
int Position;
{
	int Node, Connection;

	/* search for a connection with surface_pos == position */
	/* return the role of that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		Connection = GetConn(S, FrameId, Node);
		if (Connection != NIL) {
			if (GetSurfPos( S,Connection) == Position)
				return( GetRole( S,Connection));
			}
		}
	return( NO_FILLER);
}

int GetModeOfPosition(S, FrameId, Position)
struct SEMANTIC_NETWORK *S;
int FrameId;
int Position;
{
	int Node, Connection;

	/* search for a connection with surface_pos == position */
	/* return the semantic mode of that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		Connection = GetConn(S, FrameId, Node);
		if (Connection != NIL) {
			if (GetSurfPos( S,Connection) == Position)
				return( GetMode( S,Connection));
			}
		}
	return( NO_FILLER);
}


/**************************************************************************/
/*                                                                        */
/*                 Paradigmatic Frame Manipulation Functions              */
/*                                                                        */
/**************************************************************************/
/* returns the term playing the POSITIVE role in a paradigmatic frame */
int GetPositiveTerm( S, FrameId)
struct SEMANTIC_NETWORK *S;
int FrameId;
{
	int Node, Connection;

	if (not(GetType( S,FrameId) == PFRAME)) {
		printf("ERROR:Seeking POSITIVE role for non-paradigmatic frame.\n");
		exit(ERROR_VALUE);
		}
	/* search for a connection with POSITIVE as its role */
	/* return the verb at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		if (GetConn(S, FrameId, Node) != NIL) {
			Connection = GetConn(S, FrameId, Node);
			if (GetRole( S,Connection) == POSITIVE) return( Node);
			}
		}
	return( NO_FILLER);
}

/* returns the term playing the NEGATIVE role in a paradigmatic frame */
int GetNegativeTerm( S, FrameId)
struct SEMANTIC_NETWORK *S;
int FrameId;
{
	int Node, Connection;

	if (not(GetType( S,FrameId) == PFRAME)) {
		printf("ERROR:Seeking NEGATIVE role for non-paradigmatic frame.\n");
		exit(ERROR_VALUE);
		}
	/* search for a connection with NEGATIVE as its role */
	/* return the verb at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		if (GetConn(S, FrameId, Node) != NIL) {
			Connection = GetConn(S, FrameId, Node);
			if (GetRole( S,Connection) == NEGATIVE) return( Node);
			}
		}
	return( NO_FILLER);
}

/* determines whether x and y are opposites */
int Opposites( S, x, y)
struct SEMANTIC_NETWORK *S;
int x;
int y;
{
	int node;
	int PosTerm, NegTerm;

	for (node = 0; node < GetNumNodes( S); node++){
		if (GetType( S,node) == PFRAME) {
			PosTerm = GetPositiveTerm(S, node);
			NegTerm = GetNegativeTerm(S, node);
			if (((x == PosTerm) && (y == NegTerm))
				|| ((x == NegTerm) && (y == PosTerm)))
					return( TRUE);
			}
		}
	return( FALSE);
}

/* finds the opposite of a term */
int FindOpposite(S, term)
struct SEMANTIC_NETWORK *S;
int term;
{
	cons_ptr PosFrames, NegFrames;
	int TheFrame;
	cons_ptr OppTerms;
	int opposite;

	/* find paradigmatic frames in which term plays positive role */
	PosFrames = NIL;
	PosFrames = node_trans(term, S, POSITIVE, LITERAL, PosFrames);
/*  printf("The para frames in which <%s> is POSITIVE:\n",
			GetName( S,term));
	print_node_names(PosFrames, S);*/

	/* find paradigmatic frames in which term plays negative role */
	NegFrames = NIL;
	NegFrames = node_trans(term, S, NEGATIVE, LITERAL, NegFrames);
/*  printf("The para frames in which <%s> is NEGATIVE:\n",
			GetName( S,term));
	print_node_names(NegFrames, S);*/

	if ((length(PosFrames) == 1) && (length(NegFrames) == 0)) {
		/* then term is positive term in one para frame */
		/* opposite is the negative term of that frame */
		TheFrame = car(PosFrames);
		OppTerms = NIL;
		OppTerms = node_trans(TheFrame, S, NEGATIVE, LITERAL, OppTerms);
		opposite = car(OppTerms);
		}
	else if ((length(PosFrames) == 0) && (length(NegFrames) == 1)) {
		/* then term is negative term in one para frame */
		/* opposite is the positive term of that frame */
		TheFrame = car(NegFrames);
		OppTerms = NIL;
		OppTerms = node_trans(TheFrame, S, POSITIVE, LITERAL, OppTerms);
		opposite = car(OppTerms);
		}
	else {
		printf("ERROR: <%s> has more than one opposite.\n",
				GetName( S,term));
		exit(ERROR_VALUE);
		}

	return(opposite);
}


/**************************************************************************/
/*                                                                        */
/*                        Rule Manipulation Functions                     */
/*                                                                        */
/**************************************************************************/
/* returns the term playing the ANTECEDENT role in a rule */
int GetAntecedent( S, FrameId)
struct SEMANTIC_NETWORK *S;
int FrameId;
{
	int Node, Connection;

	if (not(GetType( S,FrameId) == RULE)) {
		printf("ERROR:Seeking ANTECEDENT role for non-rule <%s>.\n",
				GetName( S,FrameId));
		exit(ERROR_VALUE);
		}
	/* search for a connection with ANTECEDENT as its role */
	/* return the verb at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		if (GetConn(S, FrameId, Node) != NIL) {
			Connection = GetConn(S, FrameId, Node);
			if (GetRole( S,Connection) == ANTECEDENT) return( Node);
			}
		}
	return( NO_FILLER);
}

/* returns the term playing the CONSEQUENT role in a rule */
int GetConsequent( S, FrameId)
struct SEMANTIC_NETWORK *S;
int FrameId;
{
	int Node, Connection;

	if (not(GetType( S,FrameId) == RULE)) {
		printf("ERROR:Seeking CONSEQUENT role for non-rule <%s>.\n",
				GetName( S,FrameId));
		exit(ERROR_VALUE);
		}
	/* search for a connection with CONSEQUENT as its role */
	/* return the rule at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		if (GetConn(S, FrameId, Node) != NIL) {
			Connection = GetConn(S, FrameId, Node);
			if (GetRole( S,Connection) == CONSEQUENT) return( Node);
			}
		}
	return( NO_FILLER);
}


/* returns all the consequents of a rule */
cons_ptr GetAllConsequentFrames( S, ThisRule, AllConsequents)
struct SEMANTIC_NETWORK *S;
int ThisRule;
cons_ptr AllConsequents;
{
	int TheConsequent;
	cons_ptr TheBranches;

	TheConsequent = GetConsequent(S, ThisRule);
	switch (GetType( S,TheConsequent)) {
		case SFRAME:
			AllConsequents = cons(TheConsequent, AllConsequents);
			break;
		case CONJUNCTION:
			TheBranches = NIL;
			TheBranches = node_trans(TheConsequent, S,
							ANDBRANCH, LITERAL, TheBranches);
			AllConsequents = list_union(TheBranches, AllConsequents);
			break;
		default:
			_clearscreen( _GCLEARSCREEN);
			printf("ERROR:Consequent of rule <%s> is bad grammar category.",
				GetName( S,ThisRule));
			exit(ERROR_VALUE);
			break;
		}
	return( AllConsequents);
}


/**************************************************************************/
/*                                                                        */
/*                      Property Manipulation Functions                   */
/*                                                                        */
/**************************************************************************/
/* returns the term playing the ATTRIBUTE role in a property frame */
int GetAttributeTerm( S, FrameId)
struct SEMANTIC_NETWORK *S;
int FrameId;
{
	int Node, Connection;

	if (not(GetType( S,FrameId) == PROPERTYTYPE)) {
		printf("ERROR:Seeking ATTRIBUTE role for non-property frame.\n");
		exit(ERROR_VALUE);
		}
	/* search for a connection with ATTRIBUTE as its role */
	/* return the verb at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		if (GetConn(S, FrameId, Node) != NIL) {
			Connection = GetConn(S, FrameId, Node);
			if (GetRole( S,Connection) == ATTRIBUTE) return( Node);
			}
		}
	return( NO_FILLER);
}

/* returns the term playing the OBJECT role in a property frame */
int GetObjectTerm( S, FrameId)
struct SEMANTIC_NETWORK *S;
int FrameId;
{
	int Node, Connection;

	if (not(GetType( S,FrameId) == PROPERTYTYPE)) {
		printf("ERROR:Seeking OBJECT role for non-property frame.\n");
		exit(ERROR_VALUE);
		}
	/* search for a connection with OBJECT as its role */
	/* return the verb at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from FrameId to Node */
		if (GetConn(S, FrameId, Node) != NIL) {
			Connection = GetConn(S, FrameId, Node);
			if (GetRole( S,Connection) == OBJECT) return( Node);
			}
		}
	return( NO_FILLER);
}


/* returns the terms that are LITERAL VALUES of a property frame */
cons_ptr GetLiteralValues( S, FrameId, Values)
struct SEMANTIC_NETWORK *S;
int FrameId;
cons_ptr Values;
{
	if (not(GetType( S,FrameId) == PROPERTYTYPE)) {
		printf("ERROR:Seeking VALUES for non-property frame.\n");
		exit(ERROR_VALUE);
		}
	Values = node_trans( FrameId, S, VALUE, LITERAL, Values);
	return( Values);
}

/* returns the terms that are ANALOGICAL VALUES of a property frame */
cons_ptr GetAnalogicalValues( S, FrameId, AnaValues)
struct SEMANTIC_NETWORK *S;
int FrameId;
cons_ptr AnaValues;
{
	if (not(GetType( S,FrameId) == PROPERTYTYPE)) {
		printf("ERROR:Seeking VALUES for non-property frame.\n");
		exit(ERROR_VALUE);
		}
	AnaValues = node_trans( FrameId, S, VALUE, ANALOGICAL, AnaValues);
	return( AnaValues);
}

/* returns the PREDECESSOR of a term */
int GetPredecessorOf( S, term)
struct SEMANTIC_NETWORK *S;
int term;
{
	int Node, Connection;

	/* search for a connection with PREDECESSOR as its role */
	/* return the term at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from term to Node */
		if (GetConn(S, term, Node) != NIL) {
			Connection = GetConn(S, term, Node);
			if (GetRole( S,Connection) == PREDECESSOR) return( Node);
			}
		}
	return( NO_FILLER);
}

/* returns the SUCCESSOR of a term */
int GetSuccessorOf( S, term)
struct SEMANTIC_NETWORK *S;
int term;
{
	int Node, Connection;

	/* search for a connection with SUCCESSOR as its role */
	/* return the term at that connection or NO_FILLER */
	for (Node = 0; Node < GetNumNodes( S); Node++) {
		/* if there is a connection from term to Node */
		if (GetConn(S, term, Node) != NIL) {
			Connection = GetConn(S, term, Node);
			if (GetRole( S,Connection) == SUCCESSOR) return( Node);
			}
		}
	return( NO_FILLER);
}


/**************************************************************************/
/*                                                                        */
/*                      Feature Manipulation Functions                    */
/*                                                                        */
/**************************************************************************/
cons_ptr GetFeatures( S, Noun, Mode, TheFeatures)
struct SEMANTIC_NETWORK *S;
int Noun;   /* find the features of this Noun */
int Mode;   /* the semantic mode */
cons_ptr TheFeatures;
{
	cons_ptr MFrames, FeaturesOfFrame, ptr;
	int ThisFrame;

	MFrames = NIL;
	MFrames = node_trans(Noun, S, SUBJECT, Mode, MFrames);
	ListIterator( ptr, MFrames) {
		ThisFrame = car(ptr);
		FeaturesOfFrame = NIL;
		FeaturesOfFrame = node_trans(ThisFrame, S,
									PREDICATE, Mode, FeaturesOfFrame);
		TheFeatures = cons(car(FeaturesOfFrame), TheFeatures);
		}
	return( TheFeatures);
}

/* find all the adjectives that are PREDICABLE of a noun */
cons_ptr GetPredicableAdjectives(S, Noun, AdjectivesOfNoun)
struct SEMANTIC_NETWORK *S;
int Noun;
cons_ptr AdjectivesOfNoun;
{
	cons_ptr FeaturesOfNoun;
	cons_ptr PropertiesOfNoun;
	cons_ptr ptr, Values, AllValues;
	int ThisProperty;

	/* find the adjectives predicated as FEATURES of the noun */
	FeaturesOfNoun = NIL;
	FeaturesOfNoun = GetFeatures( S, Noun, LITERAL, FeaturesOfNoun);

	/* find the properties of the noun */
	PropertiesOfNoun = NIL;
	PropertiesOfNoun = node_trans(Noun, S, PROPERTY,
									 LITERAL, PropertiesOfNoun);
	/* for each property, union the values */
	AllValues = NIL;
	ListIterator( ptr, PropertiesOfNoun) {
		ThisProperty = car(ptr);
		Values = NIL;
		Values = GetLiteralValues(S, ThisProperty, Values);
		AllValues = list_union(Values, AllValues);
		}

	AdjectivesOfNoun = list_union(FeaturesOfNoun, AllValues);
	return( AdjectivesOfNoun);
}

